<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Table;

use CBLib\Application\Application;
use CB\Plugin\GroupJive\CBGroupJive;
use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class UserTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var int  */
	public $group			=	null;
	/** @var int  */
	public $status			=	null;
	/** @var string  */
	public $date			=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__groupjive_users';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		if ( ! $this->getInt( 'group', 0 ) ) {
			$this->setError( CBTxt::T( 'Group not specified!' ) );

			return false;
		}

		$group	=	$this->group();

		if ( ! $group->getInt( 'id', 0 ) ) {
			$this->setError( CBTxt::T( 'Group does not exist!' ) );

			return false;
		}

		if ( ( $this->getInt( 'status', 0 ) !== 4 ) && ( $this->getInt( 'user_id', 0 ) === $group->getInt( 'user_id', 0 ) ) ) {
			$this->setError( CBTxt::T( 'Group owner can not be demoted!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new				=	( ! $this->getInt( 'id', 0 ) );
		$old				=	new self();

		$this->set( 'date', $this->getString( 'date', Application::Database()->getUtcDateTime() ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gj_onBeforeUpdateUser', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onBeforeCreateUser', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		// Promote to group owner if status is changed to owner:
		if ( ( $this->getInt( 'status', 0 ) === 4 )
			 && $this->group()->getInt( 'id', 0 )
			 && ( $this->group()->getInt( 'user_id', 0 ) !== $this->getInt( 'user_id', 0 ) )
		) {
			$this->group()->set( 'user_id', $this->getInt( 'user_id', 0 ) );

			$this->group()->store();
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gj_onAfterUpdateUser', array( $this, $old ) );
		} elseif ( $this->group()->getInt( 'type', 0 ) !== 4 ) {
			// Set the default notifications:
			$notification	=	new NotificationTable();

			$notification->load( array( 'user_id' => $this->getInt( 'user_id', 0 ), 'group' => $this->getInt( 'group', 0 ) ) );

			$notification->set( 'user_id', $this->getInt( 'user_id', 0 ) );
			$notification->set( 'group', $this->getInt( 'group', 0 ) );

			$notifications	=	$notification->params();

			$notifications->set( 'user_join', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_user_join', 0 ) );
			$notifications->set( 'user_leave', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_user_leave', 0 ) );
			$notifications->set( 'user_approve', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_user_approve', 0 ) );
			$notifications->set( 'user_cancel', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_user_cancel', 0 ) );
			$notifications->set( 'invite_accept', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_invite_accept', 0 ) );
			$notifications->set( 'invite_reject', CBGroupJive::getGlobalParams()->getInt( 'notifications_default_invite_reject', 0 ) );

			$_PLUGINS->trigger( 'gj_onAfterCreateUser', array( $this, &$notifications ) );

			$notification->set( 'params', $notifications->asJson() );

			$notification->store();
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function canDelete( $id = null )
	{
		if ( $this->group()->getInt( 'id', 0 )
			 && ( $this->group()->getInt( 'user_id', 0 ) === $this->getInt( 'user_id', 0 ) )
		) {
			$this->setError( CBTxt::T( 'Group owner can not be deleted!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		$_PLUGINS->trigger( 'gj_onBeforeDeleteUser', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		if ( ( ! CBGroupJive::isModerator( $this->getInt( 'user_id', 0 ) ) ) && CBGroupJive::getGlobalParams()->getInt( 'groups_delete', 1 ) ) {
			// Delete this users group invites (to and from):
			$query				=	'SELECT *'
								.	"\n FROM " . $this->getDbo()->NameQuote( '#__groupjive_invites' )
								.	"\n WHERE " . $this->getDbo()->NameQuote( 'group' ) . " = " . $this->getInt( 'group', 0 )
								.	"\n AND ( " . $this->getDbo()->NameQuote( 'user_id' ) . " = " . $this->getInt( 'user_id', 0 )
								.	"\n OR " . $this->getDbo()->NameQuote( 'user' ) . " = " . $this->getInt( 'user_id', 0 ) . " )";
			$this->getDbo()->setQuery( $query );
			$invites			=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\GroupJive\Table\InviteTable', array( $this->getDbo() ) );

			/** @var InviteTable[] $invites */
			foreach ( $invites as $invite ) {
				$invite->delete();
			}

			// Delete this users group notifications:
			$notification		=	new NotificationTable();

			$notification->load( array( 'user_id' => $this->getInt( 'user_id', 0 ), 'group' => $this->getInt( 'group', 0 ) ) );

			if ( $notification->getInt( 'id', 0 ) ) {
				$notification->delete();
			}
		}

		$_PLUGINS->trigger( 'gj_onAfterDeleteUser', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @return GroupTable
	 */
	public function group()
	{
		return CBGroupJive::getGroup( $this->getInt( 'group', 0 ) );
	}

	/**
	 * @return string
	 */
	public function status()
	{
		static $cache				=	array();

		$id							=	$this->getInt( 'status', 0 );

		if ( ! isset( $cache[$id] ) ) {
			switch ( $id ) {
				case -1:
					$cache[$id]		=	CBTxt::T( 'Banned' );
					break;
				case 0:
					$cache[$id]		=	CBTxt::T( 'Pending' );
					break;
				case 1:
					$cache[$id]		=	CBTxt::T( 'Member' );
					break;
				case 2:
					$cache[$id]		=	CBTxt::T( 'Moderator' );
					break;
				case 3:
					$cache[$id]		=	CBTxt::T( 'Admin' );
					break;
				case 4:
					$cache[$id]		=	CBTxt::T( 'Owner' );
					break;
				default:
					$cache[$id]		=	CBTxt::T( 'Unknown' );
					break;
			}
		}

		return $cache[$id];
	}
}