<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity\Table;

use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Comments;
use CBLib\Registry\ParamsInterface;
use CB\Plugin\Activity\CBActivity;
use CB\Plugin\Activity\Tags;
use CB\Plugin\Activity\Likes;

defined('CBLIB') or die();

class CommentTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var string  */
	public $asset			=	null;
	/** @var string  */
	public $message			=	null;
	/** @var int  */
	public $system			=	null;
	/** @var int  */
	public $published		=	null;
	/** @var int  */
	public $pinned			=	null;
	/** @var string  */
	public $date			=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_activity_comments';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		if ( trim( $this->getHtml( 'message', '' ) ) === '' ) {
			$this->setError( CBTxt::T( 'Message not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new	=	( ! $this->getInt( 'id', 0 ) );
		$old	=	new self();

		if ( ! $this->getString( 'asset' ) ) {
			$this->set( 'asset', 'profile.' . $this->getInt( 'user_id', 0 ) );
		}

		$this->set( 'published', $this->getInt( 'published', 1 ) );
		$this->set( 'date', $this->getString( 'date', Application::Database()->getUtcDateTime() ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			if ( in_array( false, $_PLUGINS->trigger( 'activity_onBeforeUpdateComment', array( &$this, $old ) ), true ) ) {
				return false;
			}
		} elseif ( in_array( false, $_PLUGINS->trigger( 'activity_onBeforeCreateComment', array( &$this ) ), true ) ) {
			return false;
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'activity_onAfterUpdateComment', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'activity_onAfterCreateComment', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		if ( in_array( false, $_PLUGINS->trigger( 'activity_onBeforeDeleteComment', array( &$this ) ), true ) ) {
			return false;
		}

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		// Deletes activity about this comment:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity' )
						.	"\n WHERE ( " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( $this->getString( 'asset' ) . '.comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " LIKE " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) . '.%' ) . " )";
		$this->getDbo()->setQuery( $query );
		$activities		=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\ActivityTable', array( $this->getDbo() ) );

		/** @var ActivityTable[] $activities */
		foreach ( $activities as $activity ) {
			$activity->delete();
		}

		// Deletes hidden entries for this comment:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity_hidden' )
						.	"\n WHERE " . $this->getDbo()->NameQuote( 'type' ) . " = " . $this->getDbo()->Quote( 'comment' )
						.	"\n AND " . $this->getDbo()->NameQuote( 'object' ) . " = " . $this->getInt( 'id', 0 );
		$this->getDbo()->setQuery( $query );
		$hidden			=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\HiddenTable', array( $this->getDbo() ) );

		/** @var HiddenTable[] $hidden */
		foreach ( $hidden as $hide ) {
			$hide->delete();
		}

		// Deletes comment replies:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity_comments' )
						.	"\n WHERE ( " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( $this->getString( 'asset' ) . '.comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " LIKE " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) . '.%' ) . " )";
		$this->getDbo()->setQuery( $query );
		$replies		=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\CommentTable', array( $this->getDbo() ) );

		/** @var CommentTable[] $replies */
		foreach ( $replies as $reply ) {
			$reply->delete();
		}

		// Deletes comment specific notifications:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity_notifications' )
						.	"\n WHERE ( " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( $this->getString( 'asset' ) . '.comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " LIKE " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) . '.%' ) . " )";
		$this->getDbo()->setQuery( $query );
		$notifications	=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\NotificationTable', array( $this->getDbo() ) );

		/** @var NotificationTable[] $notifications */
		foreach ( $notifications as $notification ) {
			$notification->delete();
		}

		// Deletes comment specific tags:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity_tags' )
						.	"\n WHERE ( " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( $this->getString( 'asset' ) . '.comment.' . $this->getInt( 'id', 0 ) ) . " )";
		$this->getDbo()->setQuery( $query );
		$tags			=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\TagTable', array( $this->getDbo() ) );

		/** @var TagTable[] $tags */
		foreach ( $tags as $tag ) {
			$tag->delete();
		}

		// Deletes comment specific likes:
		$query			=	'SELECT *'
						.	"\n FROM " . $this->getDbo()->NameQuote( '#__comprofiler_plugin_activity_likes' )
						.	"\n WHERE ( " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( 'comment.' . $this->getInt( 'id', 0 ) )
						.	" OR " . $this->getDbo()->NameQuote( 'asset' ) . " = " . $this->getDbo()->Quote( $this->getString( 'asset' ) . '.comment.' . $this->getInt( 'id', 0 ) ) . " )";
		$this->getDbo()->setQuery( $query );
		$likes			=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\Activity\Table\LikeTable', array( $this->getDbo() ) );

		/** @var LikeTable[] $likes */
		foreach ( $likes as $like ) {
			$like->delete();
		}

		$_PLUGINS->trigger( 'activity_onAfterDeleteComment', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @param null|Comments $stream
	 * @return null|array
	 */
	public function theme( $stream = null )
	{
		static $cache		=	array();

		if ( $stream && ( ! CBActivity::findParamOverride( $this, 'themes', false, $stream ) ) ) {
			return null;
		}

		$themes				=	CBActivity::loadThemeOptions( true, $this->getInt( 'user_id', Application::MyUser()->getUserId() ), $stream );
		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$themeId		=	$this->params()->getInt( 'theme', 0 );
			$return			=	null;

			if ( $themeId && isset( $themes[$themeId] ) ) {
				$return		=	array( 'background' => $themes[$themeId]->background(), 'class' => $themes[$themeId]->getString( 'class' ) );
			}

			$cache[$id]		=	$return;
		}

		return $cache[$id];
	}

	/**
	 * @param null|Comments $stream
	 * @return null|string
	 */
	public function action( $stream = null )
	{
		static $cache				=	array();

		if ( $stream && ( ! CBActivity::findParamOverride( $this, 'actions', false, $stream ) ) ) {
			return null;
		}

		$actions					=	CBActivity::loadActionOptions( true, $this->getInt( 'user_id', Application::MyUser()->getUserId() ), $stream );
		$emotes						=	CBActivity::loadEmoteOptions( false, true, $this->getInt( 'user_id', Application::MyUser()->getUserId() ) );
		$id							=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$action					=	$this->params()->subTree( 'action' );
			$actionId				=	$action->getInt( 'id', 0 );
			$return					=	null;

			if ( $actionId && isset( $actions[$actionId] ) ) {
				$actionTitle		=	trim( CBTxt::T( $actions[$actionId]->getHtml( 'title' ) ) );
				$actionMessage		=	$action->getString( 'message' );

				if ( $actionMessage ) {
					$actionMessage	=	'<span class="commentActionMessage">' . trim( htmlspecialchars( $actionMessage ) ) . '</span>';
				}

				$emoteId			=	$action->getInt( 'emote', 0 );
				$actionEmote		=	null;

				if ( $emoteId && isset( $emotes[$emoteId] ) ) {
					$actionEmote	=	$emotes[$emoteId]->icon();
				}

				if ( $actionMessage ) {
					$return			=	CBTxt::T( 'COMMENT_ACTION', '[title] [message] [emote]', array( '[title]' => $actionTitle, '[message]' => $actionMessage, '[emote]' => $actionEmote ) );
				}
			}

			$cache[$id]				=	$return;
		}

		return $cache[$id];
	}

	/**
	 * @param null|Comments $stream
	 * @return null|string
	 */
	public function location( $stream = null )
	{
		static $cache						=	array();

		if ( $stream && ( ! CBActivity::findParamOverride( $this, 'locations', false, $stream ) ) ) {
			return null;
		}

		$locations							=	CBActivity::loadLocationOptions( true, $this->getInt( 'user_id', Application::MyUser()->getUserId() ), $stream );
		$id									=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$location						=	$this->params()->subTree( 'location' );
			$locationId						=	$location->getInt( 'id', 0 );
			$return							=	null;

			if ( $locationId && isset( $locations[$locationId] ) ) {
				$locationTitle				=	trim( CBTxt::T( $locations[$locationId]->getHtml( 'title' ) ) );

				$place						=	$location->getString( 'place' );
				$locationAddress			=	null;

				if ( $place ) {
					$address				=	$location->getString( 'address' );
					$url					=	$locations[$locationId]->getString( 'url' );
					$addressUrl				=	null;

					if ( ! $url ) {
						if ( $address ) {
							$addressUrl		=	'https://www.google.com/maps/place/' . urlencode( $address );
						} else {
							$addressUrl		=	'https://www.google.com/maps/search/' . urlencode( $place );
						}
					} elseif ( $url !== 'none' ) {
						$addressUrl			=	str_replace( array( '[place]', '[address]' ), array( urlencode( $place ), urlencode( ( $address ? $address : $place ) ) ), $url );
					}

					if ( $addressUrl ) {
						$addressUrl			=	'<a href="' . htmlspecialchars( $addressUrl ) . '" target="_blank" rel="nofollow noopener">' . trim( htmlspecialchars( $place ) ) . '</a>';
					} else {
						$addressUrl			=	trim( htmlspecialchars( $place ) );
					}

					$locationAddress		=	'<span class="commentLocation">' . $addressUrl . '</span>';
				}

				if ( $locationAddress ) {
					$return					=	CBTxt::T( 'COMMENT_LOCATION', '[title] [location]', array( '[title]' => $locationTitle, '[location]' => $locationAddress ) );
				}
			}

			$cache[$id]						=	$return;
		}

		return $cache[$id];
	}

	/**
	 * @return ParamsInterface
	 */
	public function attachments()
	{
		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]		=	CBActivity::prepareAttachments( $this->params()->subTree( 'links' ) );
		}

		return $cache[$id];
	}

	/**
	 * @return UserTable|ActivityTable|CommentTable|TagTable|FollowTable|LikeTable|NotificationTable|null
	 */
	public function source()
	{
		global $_PLUGINS;

		static $cache		=	array();

		$id					=	$this->getString( 'asset' );

		if ( ! isset( $cache[$id] ) ) {
			$source			=	CBActivity::getSource( $id );

			$_PLUGINS->trigger( 'activity_onCommentSource', array( $this, &$source ) );

			$cache[$id]		=	$source;
		}

		return $cache[$id];
	}

	/**
	 * @param null|Comments $stream
	 * @return Comments
	 */
	public function replies( $stream = null )
	{
		static $cache						=	array();

		$id									=	$this->getInt( 'id', 0 );

		if ( $stream ) {
			$streamId						=	$stream->id();
		} else {
			$streamId						=	0;
		}

		if ( ! isset( $cache[$id][$streamId] ) ) {
			$repliesUser					=	CBActivity::findParamOverride( $this, 'replies_user', 'stream' );

			if ( ! $repliesUser instanceof UserTable ) {
				if ( ( $repliesUser === null ) || ( $repliesUser === '' ) ) {
					$repliesUser			=	'stream';
				}

				switch ( $repliesUser ) {
					case 'user':
						$repliesUser		=	null;
						break;
					case 'stream':
						if ( $stream ) {
							$repliesUser	=	$stream->user();
						} else {
							$repliesUser	=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						}
						break;
					case 'asset':
					case 'comment':
						$repliesUser		=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						break;
				}
			}

			$repliesStream					=	new Comments( CBActivity::findAssetOverride( 'comments', $this, $stream ), $repliesUser, ( $stream ? $stream->ns() . '_replies' : null ) );

			if ( $stream ) {
				$repliesStream->set( 'direction', 'up' );

				$repliesStream->parse( $stream->asArray(), 'replies_' );

				$parent						=	$stream->getString( 'parent' );

				if ( ! $parent ) {
					$parent					=	$this->getString( 'asset' );
				}

				$repliesStream->set( 'parent', $parent );
				$repliesStream->set( 'moderators', $stream->getRaw( 'moderators', array() ) );
			}

			$repliesStream->set( 'inline', true );
			$repliesStream->set( 'auto_update', false );
			$repliesStream->set( 'auto_load', false );

			$repliesStream->set( 'comment', $this->getInt( 'id', 0 ) );

			$cache[$id][$streamId]			=	$repliesStream;
		}

		$replies							=	$cache[$id][$streamId];

		if ( $this->getRaw( '_comments' ) === false ) {
			$replies->set( 'query', false );
		} elseif ( ( $this->getRaw( '_comments' ) !== null ) && ( $this->getRaw( '_comments' ) !== true ) ) {
			$replies->set( 'query_count', $this->getInt( '_comments', 0 ) );
		}

		return $replies;
	}

	/**
	 * @param null|Comments $stream
	 * @return Tags
	 */
	public function tags( $stream = null )
	{
		static $cache					=	array();

		$id								=	$this->getInt( 'id', 0 );

		if ( $stream ) {
			$streamId					=	$stream->id();
		} else {
			$streamId					=	0;
		}

		if ( ! isset( $cache[$id][$streamId] ) ) {
			$tagsUser					=	CBActivity::findParamOverride( $this, 'tags_user', 'comment' );

			if ( ! $tagsUser instanceof UserTable ) {
				if ( ( $tagsUser === null ) || ( $tagsUser === '' ) ) {
					$tagsUser			=	'comment';
				}

				switch ( $tagsUser ) {
					case 'user':
						$tagsUser		=	null;
						break;
					case 'stream':
						if ( $stream ) {
							$tagsUser	=	$stream->user();
						} else {
							$tagsUser	=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						}
						break;
					case 'asset':
					case 'comment':
						$tagsUser		=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						break;
				}
			}

			$tagsStream					=	new Tags( CBActivity::findAssetOverride( 'tags', $this, $stream ), $tagsUser, ( $stream ? $stream->ns() . '_tags' : null ) );

			if ( $stream ) {
				$tagsStream->parse( $stream->asArray(), 'tags_' );

				$parent					=	$stream->getString( 'parent' );

				if ( ! $parent ) {
					$parent				=	$this->getString( 'asset' );
				}

				$tagsStream->set( 'parent', $parent );
				$tagsStream->set( 'moderators', $stream->getRaw( 'moderators', array() ) );
			}

			$tagsStream->set( 'inline', true );
			$tagsStream->set( 'placeholder', CBTxt::T( 'Who are you with?' ) );

			$tagsStream->set( 'comment', $this->getInt( 'id', 0 ) );

			$cache[$id][$streamId]		=	$tagsStream;
		}

		$tags							=	$cache[$id][$streamId];

		if ( $this->getRaw( '_tags' ) === false ) {
			$tags->set( 'query', false );
		} elseif ( ( $this->getRaw( '_tags' ) !== null ) && ( $this->getRaw( '_tags' ) !== true ) ) {
			$tags->set( 'query_count', $this->getInt( '_tags', 0 ) );
		}

		return $tags;
	}

	/**
	 * @param null|Comments $stream
	 * @return Likes
	 */
	public function likes( $stream = null )
	{
		static $cache					=	array();

		$id								=	$this->getInt( 'id', 0 );

		if ( $stream ) {
			$streamId					=	$stream->id();
		} else {
			$streamId					=	0;
		}

		if ( ! isset( $cache[$id][$streamId] ) ) {
			$likesUser					=	CBActivity::findParamOverride( $this, 'likes_user', 'comment' );

			if ( ! $likesUser instanceof UserTable ) {
				if ( ( $likesUser === null ) || ( $likesUser === '' ) ) {
					$likesUser			=	'comment';
				}

				switch ( $likesUser ) {
					case 'user':
						$likesUser		=	null;
						break;
					case 'stream':
						if ( $stream ) {
							$likesUser	=	$stream->user();
						} else {
							$likesUser	=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						}
						break;
					case 'asset':
					case 'comment':
						$likesUser		=	$this->getInt( 'user_id', Application::MyUser()->getUserId() );
						break;
				}
			}

			$likesStream				=	new Likes( CBActivity::findAssetOverride( 'likes', $this, $stream ), $likesUser, ( $stream ? $stream->ns() . '_likes' : null ) );

			if ( $stream ) {
				$likesStream->parse( $stream->asArray(), 'likes_' );

				$parent					=	$stream->getString( 'parent' );

				if ( ! $parent ) {
					$parent				=	$this->getString( 'asset' );
				}

				$likesStream->set( 'parent', $parent );
				$likesStream->set( 'moderators', $stream->getRaw( 'moderators', array() ) );
			}

			$likesStream->set( 'inline', true );
			$likesStream->set( 'count', true );
			$likesStream->set( 'layout', 'simple' );

			$likesStream->set( 'comment', $this->getInt( 'id', 0 ) );

			$cache[$id][$streamId]		=	$likesStream;
		}

		$likes							=	$cache[$id][$streamId];

		if ( $this->getRaw( '_likes' ) === false ) {
			$likes->set( 'query', false );
		} elseif ( ( $this->getRaw( '_likes' ) !== null ) && ( $this->getRaw( '_likes' ) !== true ) ) {
			$likes->set( 'query_count', $this->getInt( '_likes', 0 ) );
		}

		return $likes;
	}

	/**
	 * @param Comments|null $stream
	 * @return bool
	 */
	public function isPinned( $stream = null )
	{
		if ( $this->isHidden() ) {
			return false;
		}

		if ( CBActivity::findParamOverride( $this, 'pinned', true ) === false ) {
			return false;
		}

		if ( $stream && ( ! $stream->getBool( 'pinned', true ) ) ) {
			return false;
		}

		return $this->getBool( 'pinned', false );
	}

	/**
	 * @return bool
	 */
	public function isHidden()
	{
		return ( $this->getString( '_hidden_id' ) || $this->getString( '_hidden_user' ) || $this->getString( '_hidden_asset' ) );
	}
}