<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Code\Field;

use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CB\Plugin\Code\CBCodeField;
use CBLib\Language\CBTxt;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Input\Get;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

class CodeSelectField extends \cbFieldHandler
{
	/** @var string */
	private $codeError	=	null;

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		$value								=	$user->get( $field->get( 'name', null, GetterInterface::STRING ), null, GetterInterface::RAW );

		switch ( $output ) {
			case 'html':
			case 'rss':
				$values						=	$this->_explodeCBvalues( $value );
				$options					=	$this->getOptions( $field, $user, $reason, true );
				$labels						=	array();

				if ( $field->get( 'type', null, GetterInterface::STRING ) == 'codetag' ) {
					foreach ( $values as $k => $v ) {
						foreach ( $options as $option ) {
							if ( $v != $option['value'] ) {
								continue;
							}

							if ( ! in_array( $option['text'], $labels ) ) {
								$labels[]	=	$option['text'];
							}

							continue 2;
						}

						if ( ! in_array( $v, $labels ) ) {
							// Adds custom tag values:
							$labels[]		=	$v;
						}
					}
				} else {
					foreach ( $options as $option ) {
						if ( in_array( $option['value'], $values ) && ( ! in_array( $option['text'], $labels ) ) ) {
							$labels[]		=	$option['text'];
						}
					}
				}

				$displayStyle				=	$field->params->get( 'field_display_style', 0, GetterInterface::INT );
				$return						=	null;

				if ( Application::MyUser()->isGlobalModerator() && isset( $this->codeError ) && $this->codeError ) {
					$return					.=	'<div class="alert alert-danger">' . $this->codeError . '</div>';
				}

				$return						.=	$this->formatFieldValueLayout( $this->_arrayToFormat( $field, $labels, $output, ( $displayStyle == 1 ? 'ul' : ( $displayStyle == 2 ? 'ol' : ( $displayStyle == 3 ? 'tag' : ', ' ) ) ), trim( $field->params->get( 'field_display_class', null, GetterInterface::STRING ) ) ), $reason, $field, $user );

				return $return;
			case 'htmledit':
				return $this->getEdit( $value, $field, $user, $reason, $list_compare_types );
			case 'xml':
			case 'json':
			case 'php':
			case 'csv':
				if ( ( $output != 'csv' ) && ( substr( $reason, -11 ) == ':translated' ) ) {
					if ( in_array( $field->get( 'type', null, GetterInterface::STRING ), array( 'coderadio', 'codeselect' ) ) ) {
						return $this->_formatFieldOutput( $field->get( 'name', null, GetterInterface::STRING ), CBTxt::T( $value ), $output, ( $output != 'xml' ) );
					}

					$values					=	$this->_explodeCBvalues( $value );

					foreach ( $values as $k => $v ) {
						$values[$k]			=	CBTxt::T( $v );
					}

					return $this->_arrayToFormat( $field, $values, $output );
				}

				if ( in_array( $field->get( 'type', null, GetterInterface::STRING ), array( 'coderadio', 'codeselect' ) ) ) {
					return $this->_formatFieldOutput( $field->get( 'name', null, GetterInterface::STRING ), $value, $output, ( $output != 'xml' ) );
				}

				return $this->_arrayToFormat( $field, $this->_explodeCBvalues( $value ), $output );
			case 'csvheader':
			case 'fieldslist':
			default:
				return parent::getField( $field, $user, $output, $reason, $list_compare_types );
		}
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function prepareFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		$isTags								=	( $field->get( 'type', null, GetterInterface::STRING ) == 'codetag' );

		$this->_prepareFieldMetaSave( $field, $user, $postdata, $reason );

		$options							=	$this->getOptions( $field, $user, $reason, 'value', $postdata );

		if ( ( ! $options ) && ( ! $isTags ) ) {
			// There are no options available to this field so just skip validation and storage
			return;
		}

		foreach ( $field->getTableColumns() as $col ) {
			$value							=	cbGetParam( $postdata, $col, null, _CB_ALLOWRAW );

			if ( is_array( $value ) ) {
				if ( count( $value ) > 0 ) {
					$okVals					=	array();

					foreach ( $value as $k => $v ) {
						$v					=	stripslashes( $v );

						if ( in_array( $v, $okVals ) ) {
							continue;
						}

						if ( in_array( $v, $options ) ) {
							$okVals[$k]		=	$v;
						} elseif ( $isTags ) {
							// Allow unauthorized values for tags, but clean them to strings:
							$okVals[$k]		=	Get::clean( $v, GetterInterface::STRING );
						}
					}

					$value					=	$this->_implodeCBvalues( $okVals );
				} else {
					$value					=	'';
				}
			} elseif ( ( $value === null ) || ( $value === '' ) ) {
				$value						=	'';
			} else {
				$value						=	stripslashes( $value );

				if ( ! in_array( $value, $options ) ) {
					if ( $isTags ) {
						// Allow unauthorized value for tags, but clean it to string:
						$value				=	Get::clean( $value, GetterInterface::STRING );
					} else {
						$value				=	null;
					}
				}
			}

			if ( $this->validate( $field, $user, $col, $value, $postdata, $reason ) ) {
				if ( isset( $user->$col ) && ( (string) $user->$col ) !== (string) $value ) {
					$this->_logFieldUpdate( $field, $user, $reason, $user->$col, $value );
				}
			}

			$user->$col						=	$value;
		}
	}

	/**
	 * Finder:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $searchVals          RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata            Typically $_POST (but not necessarily), filtering required.
	 * @param  int         $list_compare_types  IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @param  string      $reason              'edit' for save user edit, 'register' for save registration
	 * @return \cbSqlQueryPart[]
	 */
	public function bindSearchCriteria( &$field, &$searchVals, &$postdata, $list_compare_types, $reason )
	{
		switch ( $field->get( 'type', null, GetterInterface::STRING ) ) {
			case 'codetag':
				$fieldType				=	'tag';
				break;
			case 'codemulticheckbox':
				$fieldType				=	'multicheckbox';
				break;
			case 'coderadio':
				$fieldType				=	'radio';
				break;
			case 'codemultiselect':
				$fieldType				=	'multiselect';
				break;
			case 'codeselect':
			default:
				$fieldType				=	'select';
				break;
		}

		if ( ( $fieldType == 'radio' ) && in_array( $list_compare_types, array( 0, 2 ) ) ) {
			$fieldType					=	'multicheckbox';
		}

		$query							=	array();
		$searchMode						=	$this->_bindSearchMode( $field, $searchVals, $postdata, ( strpos( $fieldType, 'multi' ) === 0 ? 'multiplechoice' : 'singlechoice' ), $list_compare_types );

		if ( ! $searchMode ) {
			return array();
		}

		$user							=	\CBuser::getMyUserDataInstance();
		$options						=	$this->getOptions( $field, $user, $reason, 'value', $postdata );

		if ( ( ! $options ) && ( $fieldType !== 'tag' ) ) {
			// There are no options available to this field so skip searching
			return $query;
		}

		foreach ( $field->getTableColumns() as $col ) {
			$value						=	cbGetParam( $postdata, $col );

			if ( is_array( $value ) ) {
				if ( count( $value ) > 0 ) {
					foreach ( $value as $k => $v ) {
						if ( ( count( $value ) == 1 ) && ( $v === '' ) ) {
							if ( $list_compare_types == 1 ) {
								$value	=	'';		// Advanced search: "None": checked: search for nothing selected
							} else {
								$value	=	null;	// Type 0 and 2 : Simple search: "Do not care" checked: do not search
							}
							break;
						}

						$v				=	stripslashes( $v );

						if ( in_array( $v, $options ) ) {
							$value[$k]	=	$v;
						} elseif ( $fieldType == 'tag' ) {
							// Allow unauthorized values for tags, but clean them to strings:
							$value[$k]	=	Get::clean( $v, GetterInterface::STRING );
						} else {
							unset( $value[$k] );
						}
					}
				} else {
					$value				=	null;
				}

				if ( ( $value !== null ) && ( $value !== '' ) && in_array( $searchMode, array( 'is', 'isnot' ) ) ) {
					$value				=	stripslashes( $this->_implodeCBvalues( $value ) );
				}
			} else {
				if ( ( $value !== null ) && ( $value !== '' ) ) {
					if ( ! in_array( $value, $options ) ) {
						if ( $fieldType == 'tag' ) {
							// Allow unauthorized value for tags, but clean it to string:
							$value		=	Get::clean( $value, GetterInterface::STRING );
						} else {
							$value		=	null;
						}
					}
				} else {
					if ( ( $list_compare_types == 1 ) && in_array( $searchMode, array( 'is', 'isnot' ) ) ) {
						$value			=	'';
					} else {
						$value			=	null; // 'none' is not checked and no other is checked: search for DON'T CARE
					}
				}
			}

			if ( $value !== null ) {
				$searchVals->$col		=	$value;

				$sql					=	new \cbSqlQueryPart();
				$sql->tag				=	'column';
				$sql->name				=	$col;
				$sql->table				=	$field->get( 'table', null, GetterInterface::STRING );
				$sql->type				=	'sql:field';
				$sql->operator			=	'=';
				$sql->value				=	$value;
				$sql->valuetype			=	'const:string';
				$sql->searchmode		=	$searchMode;

				$query[]				=	$sql;
			}
		}

		return $query;
	}

	/**
	 * @param mixed      $value
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason             'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param int        $list_compare_types IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	private function getEdit( $value, $field, $user, $reason, $list_compare_types )
	{
		global $_CB_framework;

		$options							=	$this->getOptions( $field, $user, $reason );

		switch ( $field->get( 'type', null, GetterInterface::STRING ) ) {
			case 'codetag':
				$fieldType					=	'tag';
				break;
			case 'codemulticheckbox':
				$fieldType					=	'multicheckbox';
				break;
			case 'coderadio':
				$fieldType					=	'radio';
				break;
			case 'codemultiselect':
				$fieldType					=	'multiselect';
				break;
			case 'codeselect':
			default:
				$fieldType					=	'select';
				break;
		}

		if ( ( ! $options ) && ( $fieldType !== 'tag' ) ) {
			if ( $field->getBool( '_isAjaxUpdate', false ) ) {
				// There are no options available to this field and its an ajax response so lets let the ajax respone know we're empty
				$field->set( '_isAjaxUpdateEmpty', true );
			}

			// There's nothing to display so make the field hidden and not required, but still exist on the page for ajax plugin purposes
			$field->set( 'cssclass', 'hidden' );
			$field->set( 'required', 0 );
		}

		$return								=	null;

		if ( Application::MyUser()->isGlobalModerator() && isset( $this->codeError ) && $this->codeError ) {
			$return							.=	'<div class="alert alert-danger">' . $this->codeError . '</div>';
		}

		if ( $fieldType == 'tag' ) {
			static $loaded	=	0;

			if ( ! $loaded++ ) {
				$js							=	"$( 'select.cbCodeSelectTag' ).cbselect({"
											.		"tags: true,"
											.		"language: {"
											.			"errorLoading: function() {"
											.				"return " . json_encode( CBTxt::T( 'The results could not be loaded.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"inputTooLong: function() {"
											.				"return " . json_encode( CBTxt::T( 'Search input too long.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"inputTooShort: function() {"
											.				"return " . json_encode( CBTxt::T( 'Search input too short.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"loadingMore: function() {"
											.				"return " . json_encode( CBTxt::T( 'Loading more results...' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"maximumSelected: function() {"
											.				"return " . json_encode( CBTxt::T( 'You cannot select any more choices.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"noResults: function() {"
											.				"return " . json_encode( CBTxt::T( 'No results found.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"searching: function() {"
											.				"return " . json_encode( CBTxt::T( 'Searching...' ), JSON_HEX_TAG ) . ";"
											.			"}"
											.		"}"
											.	"});";

				$_CB_framework->outputCbJQuery( $js, 'cbselect' );
			}
		}

		if ( $reason == 'search' ) {
			$displayType					=	$fieldType;

			if ( ( $fieldType == 'radio' ) && ( ( $list_compare_types == 2 ) || ( is_array( $value ) && ( count( $value ) > 1 ) ) ) ) {
				$displayType				=	'multicheckbox';
			}

			if ( ( $fieldType == 'select' ) && ( ( $list_compare_types == 1 ) || ( is_array( $value ) && ( count( $value ) > 1 ) ) ) ) {
				$displayType				=	'multiselect';
			}

			if ( in_array( $list_compare_types, array( 0, 2 ) ) && ( ! in_array( $displayType, array( 'multicheckbox', 'tag' ) ) ) ) {
				if ( $options && ( $options[0]->value == '' ) ) {
					// About to add 'No preference' so remove custom blank
					unset( $options[0] );
				}

				array_unshift( $options, \moscomprofilerHTML::makeOption( '', CBTxt::T( 'UE_NO_PREFERENCE', 'No preference' ) ) );
			}

			$html							=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $displayType, $value, null, $options, true, null, false );
			$return							.=	$this->_fieldSearchModeHtml( $field, $user, $html, ( ( ( strpos( $displayType, 'multi' ) === 0 ) && ( ! in_array( $fieldType, array( 'coderadio', 'codeselect' ) ) ) ) || ( $displayType == 'tag' ) ? 'multiplechoice' : 'singlechoice' ), $list_compare_types );
		} else {
			if ( ( $fieldType == 'tag' ) && ( $value != '' ) ) {
				// Since we're a tag usage we can have custom values so lets see if any exist to be added to available options:
				$values						=	$this->_explodeCBvalues( $value );

				foreach ( $values as $k => $v ) {
					foreach ( $options as $option ) {
						if ( $v != $option->value ) {
							// Custom values we'll add further below:
							continue;
						}

						// Skip values that actually exist:
						continue 2;
					}

					// Add custom tags to the available values list:
					$options[]				=	\moscomprofilerHTML::makeOption( $v, $v );
				}
			}

			if ( in_array( $fieldType, array( 'multicheckbox', 'radio' ) ) && $field->params->get( 'field_edit_style', 0, GetterInterface::INT ) ) {
				$return						.=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $fieldType . 'buttons', $value, ( in_array( $fieldType, array( 'multicheckbox', 'multiselect', 'tag' ) ) ? $this->getDataAttributes( $field, $user, 'htmledit', $reason ) : null ), $options, true, null, false );
			} else {
				$return						.=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $fieldType, $value, ( in_array( $fieldType, array( 'multicheckbox', 'multiselect', 'tag' ) ) ? $this->getDataAttributes( $field, $user, 'htmledit', $reason ) : null ), $options, true, null, false );
			}
		}

		if ( $fieldType == 'tag' ) {
			// Workaround to tag JS conflict from above with core tag JS:
			$return							=	str_replace( 'cbSelectTag', 'cbCodeSelectTag', $return );
		}

		return $return;
	}

	/**
	 * @param FieldTable  $field
	 * @param UserTable   $user
	 * @param string      $reason
	 * @param bool|string $raw
	 * @param array       $post
	 * @return array
	 */
	private function getOptions( $field, $user, $reason, $raw = false, $post = array() )
	{
		global $_CB_database;

		static $cache							=	array();

		$options								=	array();
		$cacheId								=	$field->get( 'fieldid', 0, GetterInterface::INT );

		if ( ! isset( $cache[$cacheId] ) ) {
			$query								=	"SELECT " . $_CB_database->NameQuote( 'fieldtitle' ) . " AS " . $_CB_database->NameQuote( 'value' )
												.	", if ( " . $_CB_database->NameQuote( 'fieldlabel' ) . " != '', " . $_CB_database->NameQuote( 'fieldlabel' ) . ", " . $_CB_database->NameQuote( 'fieldtitle' ) . " ) AS " . $_CB_database->NameQuote( 'text' )
												.	", " . $_CB_database->NameQuote( 'fieldgroup' ) . " AS " . $_CB_database->NameQuote( 'group' )
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_field_values' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'fieldid' ) . " = " . (int) $cacheId
												.	"\n ORDER BY" . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$cache[$cacheId]					=	$_CB_database->loadObjectList();
		}

		$rows									=	$cache[$cacheId];

		if ( $rows ) {
			foreach ( $rows as $row ) {
				if ( $raw ) {
					if ( $row->group ) {
						continue;
					}

					if ( $raw === 'value' ) {
						$options[]				=	(string) $row->value;
					} elseif ( $raw === 'text' ) {
						$options[]				=	CBTxt::T( $row->text );
					} else {
						$options[]				=	array( 'value' => (string) $row->value, 'text' => CBTxt::T( $row->text ) );
					}
				} elseif ( $row->group ) {
					$options[]					=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( $row->text ) );
				} else {
					$options[]					=	\moscomprofilerHTML::makeOption( (string) $row->value, CBTxt::T( $row->text ) );
				}
			}
		}

		if ( $post ) {
			$postUser							=	clone $user;

			foreach ( $post as $k => $v ) {
				if ( ! $k ) {
					continue;
				}

				if ( is_array( $v ) ) {
					$multi						=	false;

					foreach ( $v as $kv => $cv ) {
						if ( is_numeric( $kv ) ) {
							$kv					=	(int) $kv;
						}

						if ( is_object( $cv ) || is_array( $cv ) || ( $kv && ( ! is_int( $kv ) ) ) ) {
							$multi				=	true;
						}
					}

					if ( ! $multi ) {
						$v						=	implode( '|*|', $v );
					} else {
						continue;
					}
				} elseif ( is_object( $v ) ) {
					continue;
				}

				$postUser->set( $k, Get::clean( $v, GetterInterface::STRING ) );
			}

			$tempUser							=	$postUser;
		} else {
			$tempUser							=	$user;
		}

		// Force the user object to CBuser instance to allow substitutions for non-existant users to work:
		$cbUser									=	new \CBuser();
		$cbUser->_cbuser						=	$tempUser;

		$code									=	$cbUser->replaceUserVars( $field->params->get( 'code', '', GetterInterface::RAW ), false, false, array( 'reason' => $reason ), false );

		if ( ! $code ) {
			return $options;
		}

		try {
			$rows								=	CBCodeField::outputCode( $code, $field, $tempUser, $reason );
		} catch ( \Exception $e ) {
			if ( Application::MyUser()->isGlobalModerator() ) {
				$this->codeError				=	$e->getMessage();
			}

			$rows								=	array();
		}

		if ( $rows ) {
			foreach ( $rows as $k => $v ) {
				if ( $raw ) {
					if ( is_array( $v ) ) {
						foreach ( $v as $value => $label ) {
							if ( $raw === 'value' ) {
								$options[]		=	(string) $value;
							} elseif ( $raw === 'text' ) {
								$options[]		=	CBTxt::T( $label );
							} else {
								$options[]		=	array( 'value' => (string) $value, 'text' => CBTxt::T( $label ) );
							}
						}
					} else {
						if ( $raw === 'value' ) {
							$options[]			=	(string) $k;
						} elseif ( $raw === 'text' ) {
							$options[]			=	CBTxt::T( $v );
						} else {
							$options[]			=	array( 'value' => (string) $k, 'text' => CBTxt::T( $v ) );
						}
					}
				} else {
					if ( is_array( $v ) ) {
						if ( in_array( $field->get( 'type', null, GetterInterface::STRING ), array( 'codeselect', 'codemultiselect' ) ) ) {
							$options[]			=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( $k ) );
						}

						foreach ( $v as $value => $label ) {
							$options[]			=	\moscomprofilerHTML::makeOption( (string) $value, CBTxt::T( $label ) );
						}

						if ( in_array( $field->get( 'type', null, GetterInterface::STRING ), array( 'codeselect', 'codemultiselect' ) ) ) {
							$options[]			=	\moscomprofilerHTML::makeOptGroup( null );
						}
					} else {
						$options[]				=	\moscomprofilerHTML::makeOption( (string) $k, CBTxt::T( $v ) );
					}
				}
			}
		}

		return $options;
	}
}