<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Query\Trigger;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CB\Plugin\Query\CBQueryField;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use CBuser;

\defined( 'CBLIB') or die();

class FieldTrigger extends \cbFieldHandler
{

	/**
	 * Checks if the field can be query validated
	 *
	 * @param null|FieldTable $field
	 * @param null|UserTable  $user
	 * @param string          $reason
	 * @param bool            $checkAjax
	 * @return bool
	 */
	private function canValidate( ?FieldTable $field, ?UserTable &$user, string $reason, bool $checkAjax = true ): bool
	{
		if ( ! ( $user instanceof UserTable ) ) {
			$user				=	new UserTable();
		}

		if ( ( $field instanceof FieldTable )
			 && ( ( ! Application::Application()->isClient( 'administrator' ) ) || ( Application::Cms()->getClientId() && Application::Config()->getBool( 'adminrequiredfields', true ) ) )
			 && \in_array( $reason, [ 'edit', 'register' ], true )
		) {
			if ( ! ( $field->params instanceof ParamsInterface ) ) {
				$field->params	=	new Registry( $field->params );
			}

			$readOnly			=	$field->getBool( 'readonly', false );

			if ( ( $field->getString( 'name', '' ) === 'username' ) && ( ! Application::Config()->getBool( 'usernameedit', true ) ) ) {
				$readOnly		=	true;
			}

			if ( ( ( ! $readOnly ) || ( $reason === 'register' ) || Application::Application()->isClient( 'administrator' ) )
				 && $field->params->getBool( 'qry_validate', false )
				 && ( ( $checkAjax && $field->params->getBool( 'qry_validate_ajax', false ) ) || ( ! $checkAjax ) )
			) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Checks if the field can be query auto completed
	 *
	 * @param null|FieldTable $field
	 * @param null|UserTable  $user
	 * @param string          $reason
	 * @return bool
	 */
	private function canAutoComplete( ?FieldTable $field, ?UserTable &$user, string $reason ): bool
	{
		if ( ! ( $user instanceof UserTable ) ) {
			$user				=	new UserTable();
		}

		if ( ( $field instanceof FieldTable )
			 && ( $field->getString( 'type', '' ) === 'text' )
			 && \in_array( $reason, [ 'edit', 'register' ], true )
		) {
			if ( ! ( $field->params instanceof ParamsInterface ) ) {
				$field->params	=	new Registry( $field->params );
			}

			$readOnly			=	$field->getBool( 'readonly', false );

			if ( ( $field->getString( 'name', '' ) === 'username' ) && ( ! Application::Config()->getBool( 'usernameedit', true ) ) ) {
				$readOnly		=	true;
			}

			if ( ( ( ! $readOnly ) || ( $reason === 'register' ) || Application::Application()->isClient( 'administrator' ) )
				 && $field->params->getBool( 'qry_autocomplete', false )
			) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Sends the field value through the query and tests for validity
	 *
	 * @param FieldTable  $field
	 * @param UserTable   $user
	 * @param string      $reason
	 * @param mixed       $value
	 * @param null|CBuser $cbUser
	 * @return bool
	 */
	private function queryValidate( FieldTable $field, UserTable $user, string $reason, $value, ?CBuser $cbUser = null ): bool
	{
		$tempUser					=	null;

		if ( ! $cbUser ) {
			$cbUser					=	CBuser::getInstance( $user->getInt( 'id', 0 ), false );
			$tempUser				=	$cbUser->_cbuser;

			$cbUser->_cbuser		=	$user;
		}

		$query						=	$cbUser->replaceUserVars( $field->params->getRaw( 'qry_validate_query', '' ), [ '\CB\Plugin\Query\CBQueryField', 'escapeSQL' ], false, [ 'reason' => $reason, 'value' => $value ], false );

		if ( $tempUser ) {
			$cbUser->_cbuser		=	$tempUser;
		}

		if ( ! $query ) {
			return true;
		}

		static $cache				=	[];

		$cacheId					=	md5( $query );

		if ( ! isset( $cache[$cacheId] ) ) {
			try {
				$_SQL_database		=	CBQueryField::getDatabase( $field, 'qry_validate' );

				$_SQL_database->setQuery( $query );

				$cache[$cacheId]	=	$_SQL_database->loadResultArray();
			} catch ( \Exception $e ) {
				if ( Application::MyUser()->isGlobalModerator() ) {
					$user->setError( $e->getMessage() );
				}

				$cache[$cacheId]	=	[];
			}
		}

		$results					=	$cache[$cacheId];

		if ( \count( $results ) > 0 ) {
			if ( \count( $results ) === 1 ) {
				$results			=	array_shift( $results );
			} else {
				$results			=	true;
			}
		} else {
			$results				=	false;
		}

		$validateOn					=	$field->params->getInt( 'qry_validate_on', 0 );

		if ( $results ) {
			if ( $validateOn ) {
				return true;
			}

			return false;
		}

		if ( $validateOn ) {
			return false;
		}

		return true;
	}

	/**
	 * Prepares auto complete value=>label option pairs
	 *
	 * @param string     $query
	 * @param FieldTable $field
	 * @return array
	 */
	private function queryAutoComplete( string $query, FieldTable $field ): array
	{
		if ( ! $query ) {
			return [];
		}

		$valueColumn				=	$field->params->getString( 'qry_autocomplete_col_value', '' );
		$labelColumn				=	$field->params->getString( 'qry_autocomplete_col_label', '' );
		$cacheId					=	md5( $query );

		if ( ! isset( $cache[$cacheId] ) ) {
			try {
				$_SQL_database		=	CBQueryField::getDatabase( $field, 'qry_autocomplete' );

				$_SQL_database->setQuery( $query );

				$cache[$cacheId]	=	$_SQL_database->loadAssocList();
			} catch ( \Exception $e ) {
				$cache[$cacheId]	=	[];
			}
		}

		$rows						=	$cache[$cacheId];
		$options					=	[];

		if ( $rows ) {
			foreach ( $rows as $row ) {
				if ( \is_array( $row ) ) {
					foreach ( $row as $k => $v ) {
						$row[$k]	=	trim( (string) $v );
					}
				} else {
					$row			=	trim( (string) $row );
				}

				$value				=	null;

				if ( $valueColumn && isset( $row[$valueColumn] ) ) {
					$value			=	$row[$valueColumn];
				}

				$label				=	null;

				if ( $labelColumn && isset( $row[$labelColumn] ) ) {
					$label			=	$row[$labelColumn];

					if ( ! $value ) {
						$value		=	$label;
					}
				} elseif ( $value ) {
					$label			=	$value;
				}

				if ( ( ! $value ) && ( ! $label ) ) {
					if ( \is_array( $row ) ) {
						$value		=	array_shift( $row );
					} else {
						$value		=	$row;
					}

					$label			=	$value;
				}

				if ( $value && $label ) {
					$options[]		=	[ 'value' => (string) $value, 'label' => CBTxt::T( (string) $label ) ];
				}
			}
		}

		return $options;
	}

	/**
	 * Checks if the field is empty based off fieldtype
	 * 
	 * @param FieldTable $field
	 * @param mixed      $value
	 * @return bool
	 */
	private function isEmpty( FieldTable $field, $value ): bool
	{
		if ( ( $value === '' ) || ( $value === null ) ) {
			return true;
		}

		switch ( $field->getString( 'type', '' ) ) {
			case 'fieldgroup':
				return ( (string) $value === '[]' );
			case 'date':
				return ( (string) $value === '0000-00-00' );
			case 'datetime':
				return ( (string) $value === '0000-00-00 00:00:00' );
			case 'time':
				return ( (string) $value === '00:00:00' );
			case 'integer':
			case 'points':
			case 'checkbox':
			case 'terms':
			case 'counter':
				return ( (int) $value === 0 );
			case 'rating':
			case 'float':
				return ( (float) $value === 0.0 );
		}

		return false;
	}

	/**
	 * Direct access to field for custom operations, like for Ajax
	 *
	 * WARNING: direct unchecked access, except if $user is set, then check well for the $reason ...
	 *
	 * @param null|FieldTable $field
	 * @param null|UserTable  $user
	 * @param array           $postdata
	 * @param string          $reason 'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @return string
	 */
	public function getResponse( ?FieldTable $field, ?UserTable &$user, array $postdata, string $reason ): ?string
	{
		/** @var FieldTable[] $fields */
		static $fields							=	[];

		if ( ( $this->getInput()->getString( 'function', '' ) === 'queryvalidate' ) && $this->canValidate( $field, $user, $reason ) ) {
			$addFields							=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->getString( 'qry_validate_fields', '' ) ) ) );

			if ( ! ( $user instanceof UserTable ) ) {
				$user							=	new UserTable();
			}

			$value								=	Get::get( $postdata, 'value', '', GetterInterface::RAW );

			if ( \is_array( $value ) ) {
				$value							=	implode( '|*|', $value );
			}

			$value								=	Get::clean( $value, GetterInterface::STRING );

			if ( $this->isEmpty( $field, $value ) ) {
				return null;
			}

			$cbUser								=	CBuser::getInstance( $user->getInt( 'id', 0 ), false );

			if ( $addFields ) {
				$tempUser						=	clone $user;

				foreach ( $addFields as $addField ) {
					if ( ! isset( $fields[$addField] ) ) {
						$loadField				=	new FieldTable();

						$loadField->load( (int) $addField );

						$fields[$addField]		=	$loadField;
					}

					$addFieldName				=	$fields[$addField]->getString( 'name', '' );

					if ( ! $addFieldName ) {
						continue;
					}

					$addFieldValue				=	Get::get( $postdata, $addFieldName, '', GetterInterface::RAW );

					if ( \is_array( $addFieldValue ) ) {
						$addFieldValue			=	implode( '|*|', $addFieldValue );
					}

					$tempUser->set( $addFieldName, Get::clean( $addFieldValue, GetterInterface::STRING ) );
				}

				$cbUser->_cbuser				=	$tempUser;
			}

			$valid								=	$this->queryValidate( $field, $user, $reason, $value, $cbUser );

			if ( $addFields ) {
				$cbUser->_cbuser				=	$user;
			}

			if ( $valid ) {
				$message						=	CBTxt::T( $field->params->getHtml( 'qry_validate_success', '' ), null, [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );
			} else {
				$message						=	CBTxt::T( $field->params->getHtml( 'qry_validate_error', '' ), null, [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );

				if ( ! $message ) {
					$message					=	CBTxt::T( 'QUERY_VALIDATION_ERROR', 'Not a valid input.', [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );
				}
			}

			if ( Application::MyUser()->isGlobalModerator() && $user->getError() ) {
				$message						=	$user->getError();
			}

			return json_encode( [ 'valid' => $valid, 'message' => $message ] );
		}

		if ( ( $this->getInput()->getString( 'function', '' ) === 'queryautocomplete' ) && $this->canAutoComplete( $field, $user, $reason ) ) {
			$addFields							=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->getString( 'qry_autocomplete_fields', '' ) ) ) );

			if ( ! ( $user instanceof UserTable ) ) {
				$user							=	new UserTable();
			}

			$cbUser								=	CBuser::getInstance( $user->getInt( 'id', 0 ), false );

			if ( $addFields ) {
				$tempUser						=	clone $user;

				foreach ( $addFields as $addField ) {
					if ( ! isset( $fields[$addField] ) ) {
						$loadField				=	new FieldTable();

						$loadField->load( (int) $addField );

						$fields[$addField]		=	$loadField;
					}

					$addFieldName				=	$fields[$addField]->getString( 'name', '' );

					if ( ! $addFieldName ) {
						continue;
					}

					$addFieldValue				=	Get::get( $postdata, $addFieldName, '', GetterInterface::RAW );

					if ( \is_array( $addFieldValue ) ) {
						$addFieldValue			=	implode( '|*|', $addFieldValue );
					}

					$tempUser->set( $addFieldName, Get::clean( $addFieldValue, GetterInterface::STRING ) );
				}

				$cbUser->_cbuser				=	$tempUser;
			}

			$value								=	Get::get( $postdata, 'value', '', GetterInterface::RAW );

			if ( \is_array( $value ) ) {
				$value							=	implode( '|*|', $value );
			}

			$value								=	Get::clean( $value, GetterInterface::STRING );
			$query								=	$cbUser->replaceUserVars( $field->params->getRaw( 'qry_autocomplete_query', '' ), [ '\CB\Plugin\Query\CBQueryField', 'escapeSQL' ], false, [ 'reason' => $reason, 'value' => $value ], false );

			if ( $addFields ) {
				$cbUser->_cbuser				=	$user;
			}

			if ( ! $query ) {
				return null;
			}

			return json_encode( $this->queryAutoComplete( $query, $field ) );
		}

		return null;
	}

	/**
	 * Formatter:
	 * Returns a field in specified format
	 *
	 * @param null|FieldTable $field
	 * @param null|UserTable  $user
	 * @param string          $output             'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param string          $formatting         'tr', 'td', 'div', 'span', 'none',   'table'??
	 * @param string          $reason             'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param int             $list_compare_types IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 */
	public function getDisplay( ?FieldTable $field, ?UserTable $user, string $output, string $formatting, string $reason, int $list_compare_types ): void
	{
		global $_CB_framework;

		if ( $reason === 'search' ) {
			$selectorPrefix					=	'.cbUserListSearchFields ';
		} else {
			$selectorPrefix					=	'';
		}

		if ( $this->canValidate( $field, $user, $reason ) ) {
			static $RULE_STATIC				=	0;

			if ( ! $RULE_STATIC++ ) {
				$rule						=	"params.method = 'cbqueryvalidate';"
											.	"params.data = {};"
											.	"if ( ( typeof params.fields != 'undefined' ) && ( params.fields !== null ) && $.isArray( params.fields ) ) {"
											.		"$.each( params.fields, function( index, fieldId ) {"
											.			"var fieldTarget = $( fieldId );"
											.			"if ( $( element ).closest( '.cbRepeatRow' ).length ) {"
											.				"repeatTarget = $( element ).closest( '.cbRepeatRow' ).find( fieldId );"
											.				"if ( repeatTarget.length ) {"
											.					"fieldTarget = repeatTarget;"
											.				"}"
											.			"}"
											.			"var target = null;"
											.			"if ( fieldTarget.is( 'input' ) || fieldTarget.is( 'select' ) || fieldTarget.is( 'textarea' ) ) {"
											.				"target = fieldTarget;"
											.			"} else {"
											.				"target = fieldTarget.find( 'input,select,textarea' ).not( '[name$=\"__srmch\"]' ).first();"
											.				"if ( target.is( ':checkbox' ) || target.is( ':radio' ) ) {"
											.					"target = fieldTarget.find( 'input[name=\"' + target.attr( 'name' ) + '\"]' );"
											.				"}"
											.			"}"
											.			"if ( target.length ) {"
											.				"var fieldName = target.attr( 'name' );"
											.				"if ( $( element ).closest( '.cbRepeatRow' ).length ) {"
											.					"fieldName = fieldName.replace( /^.+__\d+__/g, '' );"
											.				"}"
											.				"if ( fieldName == params.field ) {"
											.					"return true;"
											.				"}"
											.				"fieldValue = null;"
											.				"if ( target.is( 'input' ) || target.is( 'select' ) || target.is( 'textarea' ) ) {"
											.					"if ( target.is( 'input[type=\"checkbox\"]' ) || target.is( 'input[type=\"radio\"]' ) ) {"
											.						"fieldValue = [];"
											.						"target.each( function() {"
											.							"if ( $( this ).is( ':checked' ) ) {"
											.								"fieldValue.push( $( this ).val() );"
											.							"}"
											.						"});"
											.					"} else if ( target.is( 'select[multiple]' ) ) {"
											.						"fieldValue = target.val();"
											.						"if ( value && ( ! $.isArray( fieldValue ) ) ) {"
											.							"fieldValue = fieldValue.split( ',' );"
											.						"}"
											.					"} else {"
											.						"fieldValue = target.val();"
											.					"}"
											.				"}"
											.				"params.data[fieldName] = fieldValue;"
											.			"}"
											.		"});"
											.	"}"
											.	"return $.validator.methods.cbfield.call( this, value, element, params );";

				\cbValidator::addRule( 'cbqueryvalidate', $rule );
			}

			static $RULE_LOADED				=	[];

			$fieldId						=	$field->getInt( 'fieldid', 0 );

			if ( ! isset( $RULE_LOADED[$fieldId] ) ) {
				$addFields					=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->getString( 'qry_validate_fields', '' ) ) ) );
				$selectors					=	[];

				foreach ( $addFields as $addField ) {
					$selectors[]			=	$selectorPrefix . '#cbfr_' . (int) $addField;
					$selectors[]			=	$selectorPrefix . '#cbfrd_' . (int) $addField;
				}

				$jsTargets					=	[];
				$jsTargets[]				=	$selectorPrefix . '#cbfr_' . $fieldId;
				$jsTargets[]				=	$selectorPrefix . '#cbfrd_' . $fieldId;

				$js							=	"$( " . json_encode( implode( ',', $jsTargets ), JSON_HEX_TAG ) . " ).each( function() {"
											.		"var element = $( this ).find( 'input,select,textarea' ).not( '[name$=\"__srmch\"]' ).first();"
											.		"if ( element.is( ':checkbox' ) || element.is( ':radio' ) ) {"
											.			"element = $( this ).find( 'input[name=\"' + element.attr( 'name' ) + '\"]' );"
											.		"}"
											.		"if ( ! element.length ) {"
											.			"return;"
											.		"}"
											.		"element.attr({"
											.			"'data-rule-cbqueryvalidate': '" . json_encode( [ 'user' => $user->getInt( 'id', 0 ), 'field' => htmlspecialchars( $field->getString( 'name', '' ) ), 'reason' => htmlspecialchars( $reason ), 'function' => 'queryvalidate', 'fields' => $selectors ] ) . "'"
											.		"}).on( 'modified-name.cbqueryfield', function( e, oldName, newName, index ) {"
											.			"if ( oldName != newName ) {"
											.				"$( this ).removeData( 'rule-cbqueryvalidate' );"
											.				"$( this ).attr( 'data-rule-cbqueryvalidate', $( this ).attr( 'data-rule-cbqueryvalidate' ).replace( oldName.replace( '[]', '' ), newName.replace( '[]', '' ) ) );"
											.			"}"
											.		"});"
											.	"});";

				$_CB_framework->outputCbJQuery( $js );

				$RULE_LOADED[$fieldId]		=	true;
			}
		}

		if ( $this->canAutoComplete( $field, $user, $reason ) ) {
			static $COMPLETE_LOADED			=	[];

			$fieldId						=	$field->getInt( 'fieldid', 0 );

			if ( ! isset( $COMPLETE_LOADED[$fieldId] ) ) {
				$cbSpoofField				=	cbSpoofField();
				$cbSpoofString				=	cbSpoofString( null, 'fieldclass' );
				$regAntiSpamFieldName		=	cbGetRegAntiSpamFieldName();
				$regAntiSpamValues			=	cbGetRegAntiSpams();

				if ( Application::Application()->isClient( 'administrator' ) ) {
					$updateUrl				=	$_CB_framework->backendViewUrl( 'fieldclass', false, [ 'field' => $field->getString( 'name', '' ), 'function' => 'queryautocomplete', 'user' => $user->getInt( 'id', 0 ), 'reason' => $reason, $cbSpoofField => $cbSpoofString, $regAntiSpamFieldName => $regAntiSpamValues[0] ], 'raw' );
				} else {
					$updateUrl				=	$_CB_framework->viewUrl( 'fieldclass', false, [ 'field' => $field->getString( 'name', '' ), 'function' => 'queryautocomplete', 'user' => $user->getInt( 'id', 0 ), 'reason' => $reason, $cbSpoofField => $cbSpoofString, $regAntiSpamFieldName => $regAntiSpamValues[0] ], 'raw' );
				}

				$minLength					=	$field->params->getInt( 'qry_autocomplete_min', 3 );

				if ( ! $minLength ) {
					$minLength				=	1;
				}

				$addFields					=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->getString( 'qry_autocomplete_fields', '' ) ) ) );
				$selectors					=	[];

				foreach ( $addFields as $addField ) {
					$selectors[]			=	$selectorPrefix . '#cbfr_' . (int) $addField;
					$selectors[]			=	$selectorPrefix . '#cbfrd_' . (int) $addField;
				}

				$jsTargets					=	[];
				$jsTargets[]				=	$selectorPrefix . '#cbfr_' . $fieldId;
				$jsTargets[]				=	$selectorPrefix . '#cbfrd_' . $fieldId;

				$js							=	"$( " . json_encode( implode( ',', $jsTargets ), JSON_HEX_TAG ) . " ).cbqueryautocomplete({"
											.		"url: " . json_encode( $updateUrl, JSON_HEX_TAG ) . ","
											.		"length: " . $minLength . ","
											.		"strict: " . json_encode( $field->params->getBool( 'qry_autocomplete_strict', false ), JSON_HEX_TAG ) . ","
											.		"fields: " . json_encode( $selectors, JSON_HEX_TAG )
											.	"});";

				$_CB_framework->outputCbJQuery( $js, 'cbqueryautocomplete' );

				$COMPLETE_LOADED[$fieldId]	=	true;
			}
		}
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param null|FieldTable $field
	 * @param null|UserTable  $user     RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param array           $postdata Typically $_POST (but not necessarily), filtering required.
	 * @param string          $reason   'edit' for save user edit, 'register' for save registration
	 */
	public function checkValidation( ?FieldTable $field, ?UserTable $user, array &$postdata, string $reason ): void
	{
		if ( $this->canValidate( $field, $user, $reason, false ) ) {
			$value				=	cbGetParam( $postdata, $field->getString( 'name', '' ) );

			if ( \is_array( $value ) ) {
				$value			=	$this->_implodeCBvalues( $value );
			}

			$value				=	stripslashes( (string) $value );

			if ( ( ! $this->isEmpty( $field, $value ) ) && ( ! $this->queryValidate( $field, $user, $reason, $value ) ) ) {
				$message		=	CBTxt::T( $field->params->getHtml( 'qry_validate_error', '' ), null, [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );

				if ( ! $message ) {
					$message	=	CBTxt::T( 'QUERY_VALIDATION_ERROR', 'Not a valid input.', [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );
				}

				$this->_setValidationError( $field, $user, $reason, $message );
			}
		}

		if ( $this->canAutoComplete( $field, $user, $reason )
			 && ( ( ! Application::Application()->isClient( 'administrator' ) ) || ( Application::Application()->isClient( 'administrator' ) && Application::Config()->getBool( 'adminrequiredfields', true ) ) )
			 && $field->params->getBool( 'qry_autocomplete_strict', false )
		) {
			$value				=	cbGetParam( $postdata, $field->getString( 'name', '' ) );

			if ( \is_array( $value ) ) {
				$value			=	$this->_implodeCBvalues( $value );
			}

			$value				=	stripslashes( (string) $value );

			$cbUser				=	CBuser::getInstance( $user->getInt( 'id', 0 ), false );
			$tempUser			=	$cbUser->_cbuser;

			$cbUser->_cbuser	=	$user;

			$query				=	$cbUser->replaceUserVars( $field->params->getRaw( 'qry_autocomplete_query', '' ), [ '\CB\Plugin\Query\CBQueryField', 'escapeSQL' ], false, [ 'reason' => $reason, 'value' => $value ], false );

			$cbUser->_cbuser	=	$tempUser;

			if ( ! $query ) {
				return;
			}

			$rows				=	[];

			foreach ( $this->queryAutoComplete( $query, $field ) as $option ) {
				$rows[]			=	$option['value'];
			}

			if ( ( ! $this->isEmpty( $field, $value ) ) && ( ! \in_array( $value, $rows, true ) ) ) {
				$message		=	CBTxt::T( $field->params->getHtml( 'qry_autocomplete_strict_error', '' ), null, [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );

				if ( ! $message )  {
					$message	=	CBTxt::T( 'QUERY_AUTOCOMPLETE_VALIDATION_ERROR', 'Not a valid input.', [ '[title]' => CBTxt::T( $field->getHtml( 'title', '' ) ), '[value]' => $value ] );
				}

				$this->_setValidationError( $field, $user, $reason, $message );
			}
		}
	}
}