<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\TemplateChanger\Table;

use CB\Database\Table\PluginTable;
use CBLib\Application\Application;
use CBLib\Core\CBLib;
use CBLib\Database\Table\Table;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CBLib\Registry\GetterInterface;
use CBLib\Xml\SimpleXMLElement;

defined('CBLIB') or die();

class TemplateTable extends Table
{
	/** @var int  */
	public $id					=	null;
	/** @var int  */
	public $plugin				=	null;
	/** @var string  */
	public $theme				=	null;
	/** @var string  */
	public $variables			=	null;
	/** @var string  */
	public $params				=	null;

	/** @var string  */
	public $_name				=	null;
	/** @var string  */
	public $_element			=	null;
	/** @var string  */
	public $_description		=	null;
	/** @var int  */
	public $_default			=	null;
	/** @var int  */
	public $_published			=	null;
	/** @var int  */
	public $_viewaccesslevel	=	null;
	/** @var string  */
	public $_date				=	null;
	/** @var string  */
	public $_version			=	null;

	/** @var string  */
	public $_tpl_bootstrap		=	null;
	/** @var string  */
	public $_tpl_template		=	null;
	/** @var string  */
	public $_tpl_overrides		=	null;

	/** @var Registry  */
	protected $_theme			=	null;
	/** @var Registry  */
	protected $_params			=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl				=	'#__comprofiler_plugin_templates';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key			=	'id';

	/**
	 *	Loads a row from the database into $this object by primary key
	 *
	 * @param  int|array  $keys   [Optional]: Primary key value or array of primary keys to match. If not specified, the value of current key is used
	 * @return boolean            Result from the database operation
	 *
	 * @throws  \InvalidArgumentException
	 * @throws  \RuntimeException
	 * @throws  \UnexpectedValueException
	 */
	public function load( $keys = null )
	{
		global $_CB_framework, $_PLUGINS;

		$loaded					=	parent::load( $keys );
		$plugin					=	$this->plugin();

		if ( $plugin )  {
			$templateFolder		=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/templates/' . $plugin->get( 'folder', null, GetterInterface::STRING );

			$this->set( '_name', $plugin->get( 'name', null, GetterInterface::STRING ) );
			$this->set( '_element', $plugin->get( 'element', null, GetterInterface::STRING ) );
			$this->set( '_default', ( Application::Config()->get( 'templatedir', null, GetterInterface::STRING ) == $plugin->get( 'folder', null, GetterInterface::STRING ) ? 1 : 0 ) );
			$this->set( '_published', $plugin->get( 'published', 1, GetterInterface::INT ) );
			$this->set( '_viewaccesslevel', $plugin->get( 'viewaccesslevel', 1, GetterInterface::INT ) );
			$this->set( '_version', (string) $_PLUGINS->getPluginVersion( $plugin->get( 'id', 0, GetterInterface::INT ), true ) );

			if ( file_exists( $templateFolder . '/bootstrap.css' ) ) {
				$this->set( '_tpl_bootstrap', @file_get_contents( $templateFolder . '/bootstrap.css' ) );
			}

			if ( file_exists( $templateFolder . '/template.css' ) ) {
				$this->set( '_tpl_template', @file_get_contents( $templateFolder . '/template.css' ) );
			}

			if ( file_exists( $templateFolder . '/override.css' ) ) {
				$this->set( '_tpl_overrides', @file_get_contents( $templateFolder . '/override.css' ) );
			}

			$xmlFile			=	$_PLUGINS->getPluginXmlPath( $plugin );

			if ( file_exists( $xmlFile ) ) {
				try {
					$xml		=	new SimpleXMLElement( trim( file_get_contents( $xmlFile ) ) );

					if ( $xml !== null ) {
						if ( isset( $xml->description ) ) {
							$this->set( '_date', Get::clean( (string) $xml->creationDate, GetterInterface::STRING ) );
							$this->set( '_description', Get::clean( (string) $xml->description, GetterInterface::STRING ) );
						}
					}
				} catch ( \Exception $e ) {}
			}
		}

		return $loaded;
	}

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( $this->get( '_name', null, GetterInterface::STRING ) == '' ) {
			$this->setError( CBTxt::T( 'Name not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_CB_framework;

		$plugin					=	new PluginTable();

		if ( $this->get( 'plugin', 0, GetterInterface::INT ) )  {
			$plugin->load( $this->get( 'plugin', 0, GetterInterface::INT ) );
		}

		if ( ! $plugin->get( 'id', 0, GetterInterface::INT ) )  {
			$element			=	substr( md5( uniqid( rand(), true ) ), 0, 6 );

			$plugin->set( 'name', $this->get( '_name', null, GetterInterface::STRING ) );
			$plugin->set( 'element', $element );
			$plugin->set( 'type', 'templates' );
			$plugin->set( 'folder', $element );
			$plugin->set( 'viewaccesslevel', $this->get( '_viewaccesslevel', 1, GetterInterface::INT ) );
			$plugin->set( 'ordering', 99 );
			$plugin->set( 'published', $this->get( '_published', 1, GetterInterface::INT ) );
			$plugin->set( 'iscore', 0 );
			$plugin->set( 'client_id', 0 );

			if ( $plugin->getError() || ( ! $plugin->store() ) ) {
				$this->setError( $plugin->getError() );

				return false;
			}

			$this->set( 'plugin', $plugin->get( 'id', 0, GetterInterface::INT ) );
		} else {
			$plugin->set( 'name', $this->get( '_name', null, GetterInterface::STRING ) );
			$plugin->set( 'published', $this->get( '_published', 1, GetterInterface::INT ) );
			$plugin->set( 'viewaccesslevel', $this->get( '_viewaccesslevel', 1, GetterInterface::INT ) );

			if ( $plugin->getError() || ( ! $plugin->store() ) ) {
				$this->setError( $plugin->getError() );

				return false;
			}
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( $this->get( '_default', 0, GetterInterface::INT ) ) {
			$config				=	new PluginTable();

			$config->load( array( 'element' => 'cb.core' ) );

			if ( $config->get( 'id', 0, GetterInterface::INT ) )  {
				$configParams	=	new Registry( $config->get( 'params', null, GetterInterface::RAW ) );

				$configParams->set( 'templatedir', $plugin->get( 'folder', null, GetterInterface::STRING ) );

				$config->set( 'params', $configParams->asJson() );

				$config->store();
			}
		}

		$defaultFolder			=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/templates/default';
		$templateFolder			=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/templates/' . $plugin->get( 'folder', null, GetterInterface::STRING );

		if ( ! is_dir( $templateFolder ) ) {
			$oldMask			=	@umask( 0 );

			if ( @mkdir( $templateFolder, 0755, true ) ) {
				@umask( $oldMask );
				@chmod( $templateFolder, 0755 );
			} else {
				@umask( $oldMask );
			}
		}

		if ( ! file_exists( $templateFolder . '/index.html' ) ) {
			@copy( $defaultFolder . '/index.html', $templateFolder . '/index.html' );
		}

		if ( file_exists( $templateFolder . '/' . $plugin->get( 'element', null, GetterInterface::STRING ) . '.xml' ) ) {
			@unlink( $templateFolder . '/' . $plugin->get( 'element', null, GetterInterface::STRING ) . '.xml' );
		}

		$xmlFile				=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbtemplatechanger/cbtemplatechanger.xml';
		$cbVersion				=	null;

		if ( file_exists( $xmlFile ) ) {
			try {
				$xml			=	new SimpleXMLElement( trim( file_get_contents( $xmlFile ) ) );

				if ( ( $xml !== null ) && isset( $xml->version ) ) {
					$cbVersion	=	$xml->version;
				}
			} catch ( \Exception $e ) {}
		}

		if ( ! $cbVersion ) {
			$cbVersion			=	CBLib::version();
		}

		$xml					=	'<?xml version="1.0" encoding="UTF-8"?>'
								.	"\n" . '<cbinstall version="1.0.0" type="plugin" group="templates">'
								.		"\n" . '	<name>' . htmlspecialchars( $plugin->get( 'name', null, GetterInterface::STRING ) ) . '</name>'
								.		"\n" . '	<creationDate>' . htmlspecialchars( Application::Date( 'now', 'UTC' )->format( 'Y-m-d' ) ) . '</creationDate>'
								.		"\n" . '	<author>' . htmlspecialchars( ( $_CB_framework->getCfg( 'sitname' ) ? $_CB_framework->getCfg( 'sitname' ) : $_CB_framework->getCfg( 'live_site' ) ) ) . '</author>'
								.		"\n" . '	<copyright>(C) 2004-' . htmlspecialchars( Application::Date( 'now', 'UTC' )->format( 'Y' ) ) . ' ' . htmlspecialchars( $_CB_framework->getCfg( 'live_site' ) ) . ' - and its licensors, all rights reserved</copyright>'
								.		"\n" . '	<license>http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2</license>'
								.		"\n" . '	<description>' . htmlspecialchars( $this->get( '_description', null, GetterInterface::STRING ) ) . '</description>'
								.		"\n" . '	<version>' . htmlspecialchars( $cbVersion ) . '</version>'
								.		"\n" . '	<release>' . htmlspecialchars( $this->get( '_version', null, GetterInterface::STRING ) ) . '</release>'
								.		"\n" . '	<files>'
								.			"\n" . '		<filename plugin="' . $plugin->get( 'element', null, GetterInterface::STRING ) . '">template.css</filename>'
								.			"\n" . '		<filename>bootstrap.css</filename>'
								.			"\n" . '		<filename>index.html</filename>'
								.		"\n" . '	</files>'
								.	"\n" . '</cbinstall>';

		@file_put_contents( $templateFolder . '/' . $plugin->get( 'folder', null, GetterInterface::STRING ) . '.xml', $xml );

		if ( file_exists( $templateFolder . '/bootstrap.css' ) ) {
			@unlink( $templateFolder . '/bootstrap.css' );
		}

		if ( $this->_tpl_bootstrap ) {
			@file_put_contents( $templateFolder . '/bootstrap.css', $this->_tpl_bootstrap );
		}

		if ( file_exists( $templateFolder . '/template.css' ) ) {
			@unlink( $templateFolder . '/template.css' );
		}

		if ( $this->_tpl_template ) {
			@file_put_contents( $templateFolder . '/template.css', $this->_tpl_template );
		}

		if ( file_exists( $templateFolder . '/override.css' ) ) {
			@unlink( $templateFolder . '/override.css' );
		}

		if ( $this->_tpl_overrides ) {
			@file_put_contents( $templateFolder . '/override.css', $this->_tpl_overrides );
		}

		return true;
	}

	/**
	 * Copy the named array or object content into this object as vars
	 * only existing vars of object are filled.
	 * When undefined in array, object variables are kept.
	 *
	 * WARNING: DOES addslashes / escape BY DEFAULT
	 *
	 * Can be overridden or overloaded.
	 *
	 * @param  array|object  $array         The input array or object
	 * @param  string        $ignore        Fields to ignore
	 * @param  string        $prefix        Prefix for the array keys
	 * @return boolean                      TRUE: ok, FALSE: error on array binding
	 */
	public function bind( $array, $ignore = '', $prefix = null ) {
		$bind						=	parent::bind( $array, $ignore, $prefix );

		if ( $bind ) {
			// List of custom private plugin variables to parse for and bind:
			$custom					=	array( '_name', '_description', '_default', '_published', '_viewaccesslevel', '_version', '_tpl_bootstrap', '_tpl_template', '_tpl_overrides' );

			// Set the ignore variable up like bind does encase this bind call was told to ignore custom variables:
			$ignore					=	' ' . $ignore . ' ';

			foreach ( $custom as $k ) {
				// Use the same behavior as a normal bind excluding the _ ignore check for consistency:
				if ( strpos( $ignore, ' ' . $k . ' ') === false ) {
					$ak				=	$prefix . $k;

					if ( is_array( $array ) && isset( $array[$ak] ) ) {
						$this->$k	=	$array[$ak];
					} elseif ( isset( $array->$ak ) ) {
						$this->$k	=	$array->$ak;
					}
				}
			}
		}

		return $bind;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		$plugin		=	$this->plugin();

		if ( $plugin && ( ! $plugin->delete() ) )  {
			$this->setError( $plugin->getError() );

			return false;
		}

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		return true;
	}

	/**
	 * @return null|PluginTable
	 */
	public function plugin()
	{
		$pluginId				=	$this->get( 'plugin', 0, GetterInterface::INT );

		if ( ! $pluginId )  {
			return null;
		}

		static $cache			=	array();

		if ( ! isset( $cache[$pluginId] ) ) {
			$plugin				=	new PluginTable();

			$plugin->load( $pluginId );

			if ( ! $plugin->get( 'id', 0, GetterInterface::INT ) )  {
				return null;
			}

			$cache[$pluginId]	=	$plugin;
		}

		return $cache[$pluginId];
	}

	/**
	 * @return Registry
	 */
	public function theme()
	{
		if ( ! ( $this->get( '_theme', null, GetterInterface::RAW ) instanceof Registry ) ) {
			$this->set( '_theme', new Registry( $this->get( 'theme', null, GetterInterface::RAW ) ) );
		}

		return $this->get( '_theme' );
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->get( '_params', null, GetterInterface::RAW ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->get( 'params', null, GetterInterface::RAW ) ) );
		}

		return $this->get( '_params' );
	}

	/**
	 * loads the SCSS compiler JS
	 *
	 * @return null|string
	 */
	public function compiler()
	{
		global $_CB_framework, $_PLUGINS;

		$plugin			=	$_PLUGINS->getLoadedPlugin( 'user', 'cbtemplatechanger' );

		if ( ! $plugin ) {
			return null;
		}

		$params			=	new Registry();

		$params->load( $plugin->params );

		if ( ! $params->get( 'templatechanger_cdn', false, GetterInterface::BOOLEAN ) ) {
			$configUrl	=	$_CB_framework->backendViewUrl( 'editrow', true, array( 'table' => 'pluginsbrowser', 'action' => 'editrow', 'cid' => $plugin->get( 'id', 0, GetterInterface::INT ), 'cbprevstate' => base64_encode( 'option=com_comprofiler&view=showPlugins' ) ) );

			return '<div class="alert alert-info">' . CBTxt::T( 'TEMPLATE_BUILDER_CDN_NOTICE', 'You do not appear to have the SCSS Processor CDN enabled. Please enable it within <a href="[config_url]">CB Template Changer configuration</a> to generate templates. The browser based SCSS processor is delivered using a CDN to load external JavaScript files. If you would like to know more about this CDN please visit <a href="https://www.jsdelivr.com/about" target="_blank">https://www.jsdelivr.com/about</a>. This will only be used for the purposed of generating template CSS here.', array( '[config_url]' => $configUrl ) ) . '</div>';
		}

		// https://github.com/medialize/sass.js/
		// The Worker is loaded using a cross domain workaround method see: https://www.html5rocks.com/en/tutorials/workers/basics/#toc-enviornment-loadingscripts
		$_CB_framework->document->addHeadScriptUrl( 'https://cdn.jsdelivr.net/gh/medialize/sass.js/dist/sass.js' );
		// https://github.com/ai/autoprefixer-rails
		$_CB_framework->document->addHeadScriptUrl( 'https://cdn.jsdelivr.net/gh/ai/autoprefixer-rails/vendor/autoprefixer.js' );

		$root		=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbtemplatechanger/scss/';
		$files		=	new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator( $root, \FilesystemIterator::SKIP_DOTS ), \RecursiveIteratorIterator::SELF_FIRST );
		$map		=	array();

		/** @var \FilesystemIterator[] $files */
		foreach ( $files as $file ) {
			if ( $file->isDir() || ( $file->getFilename() == 'index.html' ) ) {
				continue;
			}

			$map[]	=	str_replace( str_replace( '\\', '/', $root ), '', str_replace( '\\', '/', $file->getPathname() ) );
		}

		$browsers	=	array( 'last 1 major version', '>= 1%', 'Chrome >= 45', 'Firefox >= 38', 'Edge >= 12', 'Explorer >= 10', 'iOS >= 9', 'Safari >= 9', 'Android >= 4.4', 'Opera >= 30' );
		$sample		=	'<div class="ml-2 d-inline-block border align-text-bottom cbColorPickerSample" style="width: 1.25rem; height: 1.25rem;"></div>';
		$loader		=	'<span class="cbCompileLoading mr-1 spinner-border spinner-border-sm"></span>';
		$done		=	'<div class="mt-1 text-large text-success cbCompileDone">' . CBTxt::T( 'Compiled successfully! Don\'t forget to save your template!' ) . '</div>';
		$error		=	'<div class="mt-1 text-large text-danger cbCompileError">' . addslashes( CBTxt::T( 'Failed to compile! Error: ' ) ) . "' + result.message + '</div>";

		// Sass:
		$js			=	"$.fn.findBootstrapVariable = function( variables, ignoreDefault ) {"
					.		"var name = $( this ).attr( 'name' ).replace( /theme\[([^]+)\]/ig, \"$1\" );"
					.		"var def = $( this ).attr( 'placeholder' );"
					.		"var val = $( this ).val();"
					.		"if ( ! def ) {"
					.			"var defElem = $( '.cbThemeVariables' ).find( 'input[name=\"theme[' + name + '-default]\"]' );"
					.			"if ( defElem.length ) {"
					.				"def = defElem.val();"
					.			"}"
					.		"}"
					.		"if ( def && ( def === val ) && ( ignoreDefault !== true ) ) {"
					.			"return null;"
					.		"}"
					.		"var variable = '$' + name + ': ' + val + ';';"
					.		"if ( variables.indexOf( variable ) !== -1 ) {"
					.			"return null;"
					.		"}"
					.		"if ( ( val.indexOf( '$' ) !== -1 ) && ( variables.indexOf( val ) === -1 ) ) {"
					.			"$.each( val.match( /(\\$[a-zA-Z-]+)/ig ), function( k, v ) {"
					.				"if ( v.charAt( 0 ) == '$' ) {"
					.					"var found = $( '#theme__' + v.replace( /[^a-zA-z0-9_-]+/ig, '' ) );"
					.					"if ( found.length ) {"
					.						"var dependant = found.findBootstrapVariable( variables, true );"
					.						"if ( dependant ) {"
					.							"variable = dependant + '\\n' + variable;"
					.						"}"
					.					"}"
					.				"}"
					.			"});"
					.		"}"
					.		"return variable;"
					.	"};"
					.	"$( 'textarea.cbTemplateVariables,textarea.cbTemplateBootstrap,textarea.cbTemplateTemplate,textarea.cbTemplateOverrides' ).attr( 'autocomplete', 'off' ).attr( 'autocorrect', 'off' ).attr( 'autocapitalize', 'off' ).attr( 'spellcheck', 'false' );"
					.	"var CBSCSS = new Sass( '" . addslashes( $_CB_framework->getCfg( 'live_site' ) ) . "/components/com_comprofiler/plugin/user/plug_cbtemplatechanger/js/sass.worker.js' );"
					.	"var CBSCSSMap = " . ( $map ? "['" . implode( "','", $map ) . "']" : "[]" ) . ";"
					.	"CBSCSS.options({"
					.		"style: Sass.style.expanded,"
					.		"precision: 8"
					.	"});"
					.	"CBSCSS.preloadFiles( '../scss/', '', CBSCSSMap );"
					.	"$( '.cbCompileTheme' ).on( 'click', function() {"
					.		"if ( $( this ).hasClass( 'disabled' ) ) {;"
					.			"return;"
					.		"};"
					.		"$( '.cbCompileDone,.cbCompileError' ).remove();"
					.		"$( '.cbCompileTheme,.cbCompileVariables' ).addClass( 'disabled' ).prop( 'disabled', true );"
					.		"$( '.cbCompileTheme,.cbCompileVariables' ).prepend( '" . addslashes( $loader ) . "' );"
					.		"var variables = '';"
					.		"$( '.cbThemeVariables' ).find( 'input[type=\"text\"],input[type=\"radio\"]:checked,select' ).each( function() {"
					.			"var variable = $( this ).findBootstrapVariable( variables );"
					.			"if ( ! variable ) {"
					.				"return true;"
					.			"}"
					.			"variables += ( variables ? '\\n' : '' ) + variable;"
					.		"});"
					.		"$( 'textarea.cbTemplateVariables' ).val( variables );"
					.		"CBSCSS.writeFile( 'custom.scss', variables, function( success ) {"
					.			"if ( success ) {"
					.				"CBSCSS.compile( '@import \"sources/bootstrap/functions\";\\n@import \"custom\";\\n@import \"bootstrap\";', function( result ) {"
					.					"if ( result.status == 0 ) {"
					.						"$( 'textarea.cbTemplateBootstrap' ).val( result.text ).change();"
					.						"CBSCSS.compile( '@import \"sources/bootstrap/functions\";\\n@import \"custom\";\\n@import \"template\";', function( result ) {"
					.							"$( '.cbCompileLoading' ).remove();"
					.							"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.							"if ( result.status == 0 ) {"
					.								"$( 'textarea.cbTemplateTemplate' ).val( result.text ).change();"
					.								"$( '.cbCompileTheme' ).after( '" . addslashes( $done ) . "' );"
					.							"} else {"
					.								"$( '.cbCompileTheme' ).after( '" . $error . "' );"
					.							"}"
					.						"});"
					.					"} else {"
					.						"$( '.cbCompileLoading' ).remove();"
					.						"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.						"$( '.cbCompileTheme' ).after( '" . $error . "' );"
					.					"}"
					.				"});"
					.			"} else {"
					.				"$( '.cbCompileLoading' ).remove();"
					.				"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.				"$( '.cbCompileTheme' ).after( '" . $error . "' );"
					.			"}"
					.		"});"
					.	"});"
					.	"$( '.cbCompileVariables' ).on( 'click', function() {"
					.		"if ( $( this ).hasClass( 'disabled' ) ) {;"
					.			"return;"
					.		"};"
					.		"$( '.cbCompileDone,.cbCompileError' ).remove();"
					.		"$( '.cbCompileTheme,.cbCompileVariables' ).addClass( 'disabled' ).prop( 'disabled', true );"
					.		"$( '.cbCompileTheme,.cbCompileVariables' ).prepend( '" . addslashes( $loader ) . "' );"
					.		"CBSCSS.writeFile( 'custom.scss', $( 'textarea.cbTemplateVariables' ).val(), function( success ) {"
					.			"if ( success ) {"
					.				"CBSCSS.compile( '@import \"sources/bootstrap/functions\";\\n@import \"custom\";\\n@import \"bootstrap\";', function( result ) {"
					.					"if ( result.status == 0 ) {"
					.						"$( 'textarea.cbTemplateBootstrap' ).val( result.text ).change();"
					.						"CBSCSS.compile( '@import \"sources/bootstrap/functions\";\\n@import \"custom\";\\n@import \"template\";', function( result ) {"
					.							"$( '.cbCompileLoading' ).remove();"
					.							"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.							"if ( result.status == 0 ) {"
					.								"$( 'textarea.cbTemplateTemplate' ).val( result.text ).change();"
					.								"$( '.cbCompileVariables' ).after( '" . addslashes( $done ) . "' );"
					.							"} else {"
					.								"$( '.cbCompileVariables' ).after( '" . $error . "' );"
					.							"}"
					.						"});"
					.					"} else {"
					.						"$( '.cbCompileLoading' ).remove();"
					.						"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.						"$( '.cbCompileVariables' ).after( '" . $error . "' );"
					.					"}"
					.				"});"
					.			"} else {"
					.				"$( '.cbCompileLoading' ).remove();"
					.				"$( '.cbCompileTheme,.cbCompileVariables' ).removeClass( 'disabled' ).prop( 'disabled', false );"
					.				"$( '.cbCompileVariables' ).after( '" . $error . "' );"
					.			"}"
					.		"});"
					.	"});";

		// Color Picker:
		$js			.=	"var CBSCSSTyping = null;"
					.	"$.colorpicker.parsers['SCSS'] = function( color, that ) {"
					.		"if ( ( typeof color == 'string' ) && color ) {"
					.			"if ( color.charAt( 0 ) == '$' ) {"
					.				"if ( color.length == 1 ) {"
					.					"return;"
					.				"}"
					.				"var found = $( '#theme__' + color.replace( /[^a-zA-z0-9_-]+/ig, '' ) );"
					.				"if ( found.length ) {"
					.					"return that._parseColor( found.val() );"
					.				"}"
					.			"} else {"
					.				"if ( color == that.originalColor ) {"
					.					"if ( that.scssColor ) {"
					.						"that.element.closest( 'div.cbColorPicker' ).find( '.cbColorPickerSample' ).css( 'background-color', that.scssColor );"
					.						"return that._parseColor( that.scssColor );"
					.					"}"
					.					"return;"
					.				"}"
					.				"if ( ! /[a-zA-Z-]+\(/i.test( color ) ) {"
					.					"return;"
					.				"}"
					.				"that.originalColor = color;"
					.				"$.each( color.match( /(\\$[a-zA-Z-]+)/ig ), function( k, v ) {"
					.					"if ( v.charAt( 0 ) == '$' ) {"
					.						"var found = $( '#theme__' + v.replace( /[^a-zA-z0-9_-]+/ig, '' ) );"
					.						"if ( found.length ) {"
					.							"color = color.replace( v, '#' + that._parseColor( found.val() ).toHex() );"
					.						"}"
					.					"}"
					.				"});"
					.				"if ( CBSCSSTyping != null ) {"
					.					"clearTimeout( CBSCSSTyping );"
					.				"}"
					.				"CBSCSSTyping = setTimeout( function() {"
					.					"CBSCSS.compile( '.fake { color: ' + color + '; }', function( result ) {"
					.						"if ( ! result.text ) {"
					.							"that.scssColor = '';"
					.							"that.element.closest( 'div.cbColorPicker' ).find( '.cbColorPickerSample' ).css( 'background-color', that.scssColor );"
					.							"return;"
					.						"}"
					.						"var newColor = result.text.match( /color: (.+);/i );"
					.						"if ( newColor == null ) {"
					.							"that.scssColor = '';"
					.							"that.element.closest( 'div.cbColorPicker' ).find( '.cbColorPickerSample' ).css( 'background-color', that.scssColor );"
					.							"return;"
					.						"}"
					.						"that.scssColor = newColor[1];"
					.						"that.element.closest( 'div.cbColorPicker' ).find( '.cbColorPickerSample' ).css( 'background-color', that.scssColor );"
					.					"});"
					.				"}, 400 );"
					.			"}"
					.		"}"
					.		"return;"
					.	"};"
					.	"$.colorpicker.regional['cb'] = {"
					.		"ok: '" . addslashes( CBTxt::T( 'COLORPICKER_OK', 'Ok' ) ) . "',"
					.		"cancel: '" . addslashes( CBTxt::T( 'COLORPICKER_CANCEL', 'Cancel' ) ) . "',"
					.		"hsvH: '" . addslashes( CBTxt::T( 'COLORPICKER_HSV_H', 'H' ) ) . "',"
					.		"hsvS: '" . addslashes( CBTxt::T( 'COLORPICKER_HSV_S', 'S' ) ) . "',"
					.		"hsvV: '" . addslashes( CBTxt::T( 'COLORPICKER_HSV_V', 'V' ) ) . "',"
					.		"rgbR: '" . addslashes( CBTxt::T( 'COLORPICKER_RGB_R', 'R' ) ) . "',"
					.		"rgbG: '" . addslashes( CBTxt::T( 'COLORPICKER_RGB_G', 'G' ) ) . "',"
					.		"rgbB: '" . addslashes( CBTxt::T( 'COLORPICKER_RGB_B', 'B' ) ) . "'"
					.	"};"
					.	"$( 'div.cbColorPicker' ).each( function() {"
					.		"$( this ).find( '.col-form-label' ).append( '" . addslashes( $sample ) . "' );"
					.		"$( this ).find( 'input.cbColorPicker' ).colorpicker({"
					.			"colorFormat: ['#HEX', 'VAR', '#HEX3', 'NAME'],"
					.			"altField: $( this ).find( '.cbColorPickerSample' ),"
					.			"showAnim: '',"
					.			"position: {"
					.				"my: 'left top+5',"
					.				"at: 'left bottom',"
					.				"of: 'element',"
					.				"collision: 'fit none'"
					.			"},"
					.			"regional: 'cb',"
					.			"open: function( e, picker ) {"
					.				"var dialog = $( picker.colorPicker.dialog );"
					.				"dialog.addClass( 'cb_template' );"
					.				"dialog.find( 'label' ).addClass( 'w-auto' );"
					.				"dialog.find( 'input,label' ).addClass( 'm-0' );"
					.				"dialog.find( '.ui-colorpicker-preview' ).addClass( 'mb-1 d-flex border' );"
					.				"dialog.find( '.ui-colorpicker-preview > div' ).addClass( 'w-50' );"
					.				"dialog.find( '.ui-colorpicker-hsv > div,.ui-colorpicker-rgb > div' ).addClass( 'mb-1 d-flex align-items-center' );"
					.				"dialog.find( '.ui-colorpicker-hex' ).addClass( 'd-flex align-items-center' );"
					.				"dialog.find( '.ui-colorpicker-hsv label,.ui-colorpicker-rgb label' ).addClass( 'ml-1 mr-1' );"
					.				"dialog.find( '.ui-colorpicker-padding-left.ui-colorpicker-padding-top' ).addClass( 'p-0 pl-2' );"
					.				"dialog.find( '.ui-colorpicker-unit' ).addClass( 'ml-1' );"
					.				"dialog.find( '.ui-colorpicker-hex label' ).addClass( 'mr-1' );"
					.				"dialog.find( '.ui-colorpicker-number,.ui-colorpicker-hex-input' ).addClass( 'form-control form-control-sm' );"
					.				"dialog.find( '.ui-dialog-buttonpane' ).addClass( 'm-0 mt-2 p-0 border-0' );"
					.				"dialog.find( '.ui-colorpicker-cancel' ).removeClass( 'ui-button' ).addClass( 'm-0 btn btn-secondary' );"
					.				"dialog.find( '.ui-colorpicker-ok' ).removeClass( 'ui-button' ).addClass( 'm-0 ml-1 btn btn-primary' );"
					.			"},"
					.			"select : function( e, picker ) {"
					.				"var varName = '$' + picker.colorPicker.element.attr( 'id' ).replace( 'theme__', '' );"
					.				"$( 'input.cbColorPicker' ).each( function() {"
					.					"if ( $( this ).val().indexOf( varName ) !== -1 ) {"
					.						"$( this ).colorpicker( 'setColor', $( this ).val() );"
					.					"}"
					.				"});"
					.			"}"
					.		"});"
					.	"});"
					.	"$( 'textarea.cbTemplateBootstrap,textarea.cbTemplateTemplate,textarea.cbTemplateOverrides' ).on( 'change', function() {"
					.		"$( this ).val( autoprefixer.process( $( this ).val(), {}, {overrideBrowserslist: ['" . implode( "', '", $browsers ) . "']}).css );"
					.	"});";

		$_CB_framework->outputCbJQuery( $js, 'colorpicker' );

		return null;
	}

	/**
	 * Preview a template
	 *
	 * @return null|string
	 */
	public function preview()
	{
		global $_CB_framework;

		if ( ! $this->get( 'id', 0, GetterInterface::INT ) )  {
			return CBTxt::T( 'Template not found.' );
		}

		if ( ! $this->get( '_element', null, GetterInterface::STRING ) )  {
			return CBTxt::T( 'Template plugin not found.' );
		}

		cbRedirect( $_CB_framework->userProfileUrl( Application::MyUser()->getUserId(), false, null, 'html', 0, array( 'cbtmplpreview' => $this->get( '_element', null, GetterInterface::STRING ) ) ) );

		return null;
	}

	/**
	 * Download a template
	 */
	public function download()
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! extension_loaded( 'zip' ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		if ( ! $this->get( 'id', 0, GetterInterface::INT ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$plugin						=	$this->plugin();

		if ( ! $plugin )  {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$pluginPath					=	str_replace( '\\', '/', realpath( $_PLUGINS->getPluginPath( $plugin ) ) );
		$fileName					=	$this->get( '_element', null, GetterInterface::STRING ) . '.zip';
		$filePath					=	$_CB_framework->getCfg( 'absolute_path' ) . '/cache/' . $fileName;

		if ( file_exists( $filePath ) ) {
			@unlink( $filePath );
		}

		$zip						=	new \ZipArchive();

		if ( $zip->open( $filePath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE ) === true ) {
			$files					=	new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator( $pluginPath, \FilesystemIterator::SKIP_DOTS ), \RecursiveIteratorIterator::SELF_FIRST );

			if ( $files ) {
				foreach ( $files as $file ) {
					$file			=	str_replace( '\\', '/', realpath( $file ) );

					if ( is_dir( $file ) ) {
						$zip->addEmptyDir( str_replace( $pluginPath . '/', '', $file . '/' ) );
					} elseif ( is_file( $file ) ) {
						$zip->addFile( $file, str_replace( $pluginPath . '/', '', $file ) );
					}
				}
			}

			$zip->close();
		} else {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$fileModifedTime			=	filemtime( $filePath );
		$fileModifedDate			=	Application::Date( $fileModifedTime, 'UTC' )->format( 'r', true, false );
		$fileEtag					=	md5_file( $filePath );

		if ( ! Application::Cms()->getClientId() ) {
			if ( ( isset( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) && ( strtotime( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) == $fileModifedTime ) ) || isset( $_SERVER['HTTP_IF_NONE_MATCH'] ) && ( trim( $_SERVER['HTTP_IF_NONE_MATCH'] ) == $fileEtag ) ) {
				header( 'HTTP/1.1 304 Not Modified' );
				exit();
			}
		}

		$fileSize					=	@filesize( $filePath );

		while ( @ob_end_clean() );

		if ( ini_get( 'zlib.output_compression' ) ) {
			ini_set( 'zlib.output_compression', 'Off' );
		}

		if ( function_exists( 'apache_setenv' ) ) {
			apache_setenv( 'no-gzip', '1' );
		}

		header( "Content-Type: application/zip" );
		header( 'Content-Disposition: attachment; modification-date="' . $fileModifedDate . '"; size=' . $fileSize . '; filename="' . $fileName . '";' );
		header( 'Content-Transfer-Encoding: binary' );
		header( 'Pragma: private' );
		header( 'Cache-Control: private' );
		header( "Last-Modified: $fileModifedDate" );
		header( "Etag: $fileEtag" );
		header( 'Accept-Ranges: bytes' );

		$start						=	0;
		$end						=	( $fileSize - 1 );
		$length						=	$fileSize;

		$isLarge					=	( $fileSize >= 524288 ); // Larger Than or Equal To 512kb
		$isRange					=	false;

		if ( isset( $_SERVER['HTTP_RANGE'] ) ) {
			if ( ! preg_match( '/^bytes=\d*-\d*(,\d*-\d*)*$/i', $_SERVER['HTTP_RANGE'] ) ) {
				header( 'HTTP/1.1 416 Requested Range Not Satisfiable' );
				header( "Content-Range: bytes */$fileSize" );
				exit();
			}

			list( , $range )		=	explode( '=', $_SERVER['HTTP_RANGE'], 2 );

			if ( strpos( $range, ',' ) !== false ) {
				header('HTTP/1.1 416 Requested Range Not Satisfiable');
				header("Content-Range: bytes $start-$end/$fileSize");
				exit;
			}

			if ( $range == '-' ) {
				$rangeStart			=	( $fileSize - substr( $range, 1 ) );
				$rangeEnd			=	$end;
			} else {
				$range				=	explode( '-', $range );
				$rangeStart			=	$range[0];
				$rangeEnd			=	( ( isset( $range[1] ) && is_numeric( $range[1] ) ) ? $range[1] : $fileSize );
			}

			$rangeEnd				=	( ( $rangeEnd > $end ) ? $end : $rangeEnd );

			if ( ( $rangeStart > $rangeEnd ) || ( $rangeStart > ( $fileSize - 1 ) ) || ( $rangeEnd >= $fileSize ) ) {
				header( 'HTTP/1.1 416 Requested Range Not Satisfiable' );
				header( "Content-Range: bytes $start-$end/$fileSize" );
				exit;
			}

			$start					=	$rangeStart;
			$end					=	$rangeEnd;
			$length					=	( ( $end - $start ) + 1 );

			header( 'HTTP/1.1 206 Partial Content' );
			header( "Content-Range: bytes $start-$end/$fileSize" );
			header( "Content-Length: $length" );

			$isRange				=	true;
		} else {
			header( 'HTTP/1.0 200 OK' );
			header( "Content-Length: $fileSize" );
		}

		if ( ! ini_get( 'safe_mode' ) ) {
			@set_time_limit( 0 );
		}

		if ( $isLarge || $isRange ) {
			$file					=	fopen( $filePath, 'rb' );

			if ( $file === false ) {
				header( 'HTTP/1.0 404 Not Found' );
				exit();
			}

			fseek( $file, $start );

			if ( $isLarge ) {
				$buffer				=	( 1024 * 8 );

				while ( ( ! feof( $file ) ) && ( ( $pos = ftell( $file ) ) <= $end ) ) {
					if ( ( $pos + $buffer ) > $end ) {
						$buffer		=	( ( $end - $pos ) + 1 );
					}

					echo fread( $file, $buffer );
					@ob_flush();
					flush();
				}
			} else {
				echo fread( $file, $length );
			}

			fclose( $file );
		} else {
			if ( readfile( $filePath ) === false ) {
				header( 'HTTP/1.0 404 Not Found' );
				exit();
			}
		}

		exit();
	}
}