<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Table\ActivityTable;

defined('CBLIB') or die();

/**
 * @method string getAsset()
 * @method Activity setAsset( $asset )
 * @method array getAssets()
 * @method Activity setAssets( $assets )
 * @method string getParentAsset()
 * @method Activity setParentAsset( $asset )
 * @method UserTable getUser()
 * @method Activity setUser( $user )
 * @method int|array getId()
 * @method Activity setId( $id )
 * @method array getModerators()
 * @method Activity setModerators( $moderators )
 * @method int|array getUserId()
 * @method Activity setUserId( $userId )
 * @method string|array getDate()
 * @method Activity setDate( $datetime )
 * @method string getSearch()
 * @method Activity setSearch( $search )
 * @method string getTitle()
 * @method Activity setTitle( $title )
 * @method string getMessage()
 * @method Activity setMessage( $message )
 * @method string getHidden()
 * @method Activity setHidden( $hidden )
 * @method bool getInline()
 * @method Activity setInline( $inline )
 * @method int getPublished()
 * @method Activity setPublished( $published )
 */
class Activity extends Stream implements ActivityInterface
{
	/** @var array $defaults */
	protected $defaults					=	array(	'template'									=>	null,
													'layout'									=>	'stream',
													'direction'									=>	'down',
													'auto_update'								=>	false,
													'auto_load'									=>	false,
													'global'									=>	true,
													'pinned'									=>	true,
													'filters'									=>	array(),
													'create'									=>	true,
													'create_access'								=>	2,
													'create_connected'							=>	true,
													'create_system'								=>	true,
													'message_limit'								=>	400,
													'collapsed'									=>	true,
													'paging'									=>	true,
													'paging_first_limit'						=>	15,
													'paging_limit'								=>	15,
													'parser_substitutions'						=>	false,
													'parser_emotes'								=>	true,
													'parser_reactions'							=>	true,
													'parser_hashtags'							=>	true,
													'parser_profiles'							=>	true,
													'parser_links'								=>	true,
													'parser_prepare'							=>	false,
													'parser_bbcode'								=>	false,
													'actions'									=>	true,
													'actions_message_limit'						=>	100,
													'actions_include'							=>	'0',
													'actions_exclude'							=>	'0',
													'locations'									=>	true,
													'locations_address_limit'					=>	200,
													'locations_include'							=>	'0',
													'locations_exclude'							=>	'0',
													'links'										=>	true,
													'links_file_extensions'						=>	'zip,rar,doc,pdf,txt,xls',
													'links_embedded'							=>	false,
													'links_link_limit'							=>	5,
													'tags'										=>	true,
													'likes'										=>	true,
													'likes_include'								=>	'0',
													'likes_exclude'								=>	'0',
													'themes'									=>	true,
													'themes_include'							=>	'0',
													'themes_exclude'							=>	'0',
													'comments'									=>	true,
													'comments_direction'						=>	'up',
													'comments_auto_update'						=>	false,
													'comments_auto_load'						=>	false,
													'comments_pinned'							=>	false,
													'comments_create'							=>	true,
													'comments_create_access'					=>	2,
													'comments_create_connected'					=>	true,
													'comments_create_system'					=>	false,
													'comments_message_limit'					=>	400,
													'comments_collapsed'						=>	true,
													'comments_paging'							=>	true,
													'comments_paging_first_limit'				=>	5,
													'comments_paging_limit'						=>	5,
													'comments_parser_substitutions'				=>	false,
													'comments_parser_emotes'					=>	true,
													'comments_parser_reactions'					=>	true,
													'comments_parser_hashtags'					=>	true,
													'comments_parser_profiles'					=>	true,
													'comments_parser_links'						=>	true,
													'comments_parser_prepare'					=>	false,
													'comments_parser_bbcode'					=>	false,
													'comments_actions'							=>	false,
													'comments_actions_message_limit'			=>	100,
													'comments_actions_include'					=>	'0',
													'comments_actions_exclude'					=>	'0',
													'comments_locations'						=>	false,
													'comments_locations_address_limit'			=>	200,
													'comments_locations_include'				=>	'0',
													'comments_locations_exclude'				=>	'0',
													'comments_links'							=>	false,
													'comments_links_file_extensions'			=>	'zip,rar,doc,pdf,txt,xls',
													'comments_links_embedded'					=>	false,
													'comments_links_link_limit'					=>	5,
													'comments_tags'								=>	false,
													'comments_likes'							=>	false,
													'comments_likes_include'					=>	'0',
													'comments_likes_exclude'					=>	'0',
													'comments_themes'							=>	false,
													'comments_themes_include'					=>	'0',
													'comments_themes_exclude'					=>	'0',
													'comments_replies'							=>	false,
													'comments_replies_direction'				=>	'up',
													'comments_replies_auto_update'				=>	false,
													'comments_replies_auto_load'				=>	false,
													'comments_replies_pinned'					=>	false,
													'comments_replies_create'					=>	true,
													'comments_replies_create_access'			=>	2,
													'comments_replies_create_connected'			=>	true,
													'comments_replies_create_system'			=>	false,
													'comments_replies_message_limit'			=>	400,
													'comments_replies_collapsed'				=>	false,
													'comments_replies_paging'					=>	true,
													'comments_replies_paging_first_limit'		=>	5,
													'comments_replies_paging_limit'				=>	5,
													'comments_replies_parser_substitutions'		=>	false,
													'comments_replies_parser_emotes'			=>	true,
													'comments_replies_parser_reactions'			=>	true,
													'comments_replies_parser_hashtags'			=>	true,
													'comments_replies_parser_profiles'			=>	true,
													'comments_replies_parser_links'				=>	true,
													'comments_replies_parser_prepare'			=>	false,
													'comments_replies_parser_bbcode'			=>	false,
													'comments_replies_actions'					=>	false,
													'comments_replies_actions_message_limit'	=>	100,
													'comments_replies_actions_include'			=>	'0',
													'comments_replies_actions_exclude'			=>	'0',
													'comments_replies_locations'				=>	false,
													'comments_replies_locations_address_limit'	=>	200,
													'comments_replies_locations_include'		=>	'0',
													'comments_replies_locations_exclude'		=>	'0',
													'comments_replies_links'					=>	false,
													'comments_replies_links_file_extensions'	=>	'zip,rar,doc,pdf,txt,xls',
													'comments_replies_links_embedded'			=>	false,
													'comments_replies_links_link_limit'			=>	5,
													'comments_replies_tags'						=>	false,
													'comments_replies_likes'					=>	false,
													'comments_replies_likes_include'			=>	'0',
													'comments_replies_likes_exclude'			=>	'0',
													'comments_replies_themes'					=>	false,
													'comments_replies_themes_include'			=>	'0',
													'comments_replies_themes_exclude'			=>	'0'
												);

	/** @var ActivityTable[] $loadedRows */
	protected static $loadedRows		=	array();

	/**
	 * Constructor for activity object
	 *
	 * @param null|string|array  $assets
	 * @param null|int|UserTable $user
	 * @param null|string        $namespace
	 */
	public function __construct( $assets = null, $user = null, $namespace = null )
	{
		if ( ! $assets ) {
			$assets		=	array( 'user.connections', 'following' );
		}

		$this->set( 'published', 1 );

		parent::__construct( $assets, $user, $namespace );
	}

	/**
	 * @param string $name
	 * @param array  $arguments
	 * @return self|string|int|array|null
	 */
	public function __call( $name, $arguments )
	{
		$method									=	substr( $name, 0, 3 );

		if ( in_array( $method, array( 'get', 'set' ), true ) ) {
			$variables							=	array( 'asset', 'assets', 'user', 'filters', 'filter', 'id', 'parent', 'moderators', 'user_id', 'date', 'search', 'title', 'message', 'hidden', 'inline', 'published' );
			$variable							=	strtolower( substr( $name, 3 ) );

			switch ( $variable ) {
				case 'userid':
					$variable					=	'user_id';
					break;
				case 'parentasset':
					$variable					=	'parent';
					break;
			}

			if ( in_array( $variable, $variables, true ) ) {
				switch ( $method ) {
					case 'get':
						switch ( $variable ) {
							case 'asset':
								return $this->asset();
							case 'assets':
								return $this->assets();
							case 'user':
								return $this->user();
							case 'filters':
								return $this->filters();
							case 'id':
							case 'user_id':
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getInt( $variable, 0 );
							case 'moderators':
								return $this->getRaw( $variable, array() );
							case 'inline':
							case 'hidden':
								return $this->getBool( $variable, false );
							case 'published':
							case 'filter':
								return $this->getInt( $variable, 0 );
							default:
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getString( $variable );
						}
					case 'set':
						switch ( $variable ) {
							case 'asset':
							case 'assets':
								$this->assets( ( $arguments ? $arguments[0] : null ) );
								break;
							case 'user':
								$this->user( ( $arguments ? $arguments[0] : null ) );
								break;
							default:
								$this->set( $variable, ( $arguments ? $arguments[0] : null ) );
								break;
						}

						return $this;
				}
			}
		}

		trigger_error( 'Call to undefined method ' . __CLASS__ . '::' . $name . '()', E_USER_ERROR );
	}

	/**
	 * Returns the currently active filter
	 *
	 * @return array
	 */
	public function filter()
	{
		$filter		=	$this->getInt( 'filter', 0 );

		if ( ! $filter ) {
			return array();
		}

		$filters	=	$this->filters();

		if ( ! isset( $filters[$filter] ) ) {
			return array();
		}

		return $filters[$filter];
	}

	/**
	 * Returns an array of available asset filters
	 *
	 * @return array
	 */
	public function filters()
	{
		if ( ( ! $this->getRaw( 'filters', array() ) )
			 || ( ! $this->user()->getInt( 'id', 0 ) )
			 || ( ! Application::MyUser()->getUserId() )
			 || ( $this->user()->getInt( 'id', 0 ) !== Application::MyUser()->getUserId() )
		) {
			return array();
		}

		$isModerator				=	Application::MyUser()->isGlobalModerator();
		$viewAccessLevels			=	Application::MyUser()->getAuthorisedViewLevels();

		$filters					=	array();
		$i							=	1;

		foreach ( $this->getRaw( 'filters', array() ) as $filter ) {
			if ( is_object( $filter ) ) {
				$filter				=	(array) $filter;
			}

			if ( ( ( ! isset( $filter['assets'] ) ) && ( ! isset( $filter['users'] ) ) ) || ( ! isset( $filter['filter'] ) ) ) {
				continue;
			}

			if ( ! isset( $filter['assets'] ) ) {
				$filter['assets']	=	null;
			}

			if ( ! isset( $filter['users'] ) ) {
				$filter['users']	=	null;
			}

			if ( ! isset( $filter['access'] ) ) {
				$filter['access']	=	1;
			}

			if ( ( ( ! $filter['assets'] ) && ( ! $filter['users'] ) ) || ( ! $filter['filter'] ) ) {
				continue;
			}

			if ( ( ! $isModerator ) && ( ! in_array( (int) $filter['access'], $viewAccessLevels, true ) ) ) {
				continue;
			}

			$filters[$i]			=	$filter;

			$i++;
		}

		return $filters;
	}

	/**
	 * Retrieves activity rows or row count
	 *
	 * @param string $output
	 * @return ActivityTable[]|int
	 */
	public function rows( $output = null )
	{
		global $_CB_framework, $_CB_database, $_PLUGINS;

		static $cache						=	array();

		$filter								=	$this->filter();
		$filterUsers						=	array();
		$filterAssets						=	array();

		if ( $filter ) {
			if ( $filter['assets'] ) {
				$filterAssets				=	$this->assets( explode( ',', $filter['assets'] ), true );
			}

			if ( $filter['users'] ) {
				$extras						=	array(	'displayed_id'	=>	$_CB_framework->displayedUser(),
														'viewer_id'		=>	Application::MyUser()->getUserId()
													);

				$filterUsers				=	explode( ',', \CBuser::getInstance( $this->user()->getInt( 'id', 0 ), false )->replaceUserVars( $filter['users'], true, false, $extras, false ) );
			}
		}

		$myId								=	Application::MyUser()->getUserId();
		$id									=	$this->getRaw( 'id' );
		$hasId								=	( ( ( $id !== null ) && ( $id !== '' ) ) || ( is_array( $id ) && $id ) );

		if ( $this->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit						=	$this->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit						=	$this->getInt( 'paging_limit', 15 );
		}

		$paging								=	( ( ! $hasId ) && $pageLimit && ( $output !== 'all' ) );
		$select								=	array();
		$join								=	array();
		$where								=	array();

		if ( $output === 'count' ) {
			$select[]						=	'COUNT(*)';
		} else {
			$select[]						=	'a.*';
		}

		if ( $hasId ) {
			if ( is_array( $this->getRaw( 'id' ) ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $id );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " = " . (int) $id;
			}
		}

		$userId								=	( $filterUsers ? $filterUsers : $this->getRaw( 'user_id' ) );

		if ( ( ( $userId !== null ) && ( $userId !== '' ) ) || ( is_array( $userId ) && $userId ) ) {
			if ( is_array( $userId ) ) {
				$connUsers					=	false;

				if ( in_array( 'connections', $userId, true ) ) {
					unset( $userId[array_search( 'connections', $userId, true )] );

					$connections			=	CBActivity::getConnections( $this->user()->getInt( 'id', 0 ) );

					if ( $connections ) {
						if ( count( $connections ) > 50 ) {
							$connUsers		=	true;
						} else {
							foreach( $connections as $connection ) {
								/** @var array $userId */
								$userId[]	=	$connection;
							}

							$userId			=	array_unique( $userId );
						}
					}
				}

				$usersWhere					=	array();

				if ( $userId ) {
					$usersWhere[]			=	"a." . $_CB_database->NameQuote( 'user_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userId );
				}

				if ( $connUsers ) {
					$usersWhere[]			=	"EXISTS ( SELECT 1"
											.	" FROM " . $_CB_database->NameQuote( '#__comprofiler_members' ) . " AS m"
											.	" LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS mcb"
											.	" ON mcb." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
											.	" LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS mj"
											.	" ON mj." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
											.	" WHERE mcb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
											.	" AND mcb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	" AND mj." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	" AND m." . $_CB_database->NameQuote( 'referenceid' ) . " = ". $this->user()->getInt( 'id', 0 )
											.	" AND m." . $_CB_database->NameQuote( 'memberid' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	" AND m." . $_CB_database->NameQuote( 'accepted' ) . " = 1 )";
				}

				$where[]					=	( count( $usersWhere ) > 1 ? "( " . implode( " OR ", $usersWhere ) . " )" : $usersWhere[0] );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			}
		}

		$assets								=	( $filterAssets ? $filterAssets : $this->assets() );

		if ( $assets && ( ! in_array( 'all', $assets, true ) ) )  {
			$queryAssets					=	$this->queryAssets( $filterAssets );

			if ( $queryAssets['assets'] || ( ( ! $userId ) && $queryAssets['users'] ) || $queryAssets['wildcards'] || $queryAssets['exists'] ) {
				$assetsWhere				=	array();

				if ( ( ( ! $userId ) && $queryAssets['users'] ) ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'user_id' ) . ( count( $queryAssets['users'] ) > 1 ? " IN " . $_CB_database->safeArrayOfIntegers( $queryAssets['users'] ) : " = " . (int) $queryAssets['users'][0] );
				}

				if ( $queryAssets['assets'] ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'asset' ) . ( count( $queryAssets['assets'] ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets['assets'] ) : " = " . $_CB_database->Quote( $queryAssets['assets'][0] ) );
				}

				if ( $queryAssets['wildcards'] ) {
					foreach ( $queryAssets['wildcards'] as $wildcard ) {
						$assetsWhere[]		=	"a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $wildcard );
					}
				}

				if ( $this->getBool( 'global', true ) && ( ! $filterAssets ) && ( ! $filterUsers ) ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'global' ) . " = 1";
				}

				if ( $queryAssets['exists'] ) {
					foreach ( $queryAssets['exists'] as $exist ) {
						$assetsWhere[]		=	"EXISTS ( " . $exist . " )";
					}
				}

				$where[]					=	( count( $assetsWhere ) > 1 ? "( " . implode( " OR ", $assetsWhere ) . " )" : $assetsWhere[0] );
			} elseif ( $output === 'count' ) {
				return 0;
			} else {
				return array();
			}
		}

		if ( ! $hasId ) {
			$date							=	$this->getRaw( 'date' );

			if ( is_array( $date ) ) {
				if ( count( $date ) > 1 ) {
					if ( ! in_array( $date[1], array( '=', '<>', '<', '>', '<=', '>=', 'REGEXP', 'NOT REGEXP', 'LIKE', 'NOT LIKE' ), true ) ) {
						$date[1]			=	'>';
					}

					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " " . $date[1] . " " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " > " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				}
			} elseif ( $date ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'date' ) . " = " . $_CB_database->Quote( ( is_int( $date ) ? Application::Database()->getUtcDateTime( $date ) : $date ) );
			}

			if ( $this->getString( 'title' ) ) {
				if ( strpos( $this->getString( 'title' ), '%' ) !== false ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'title' ) . " LIKE " . $_CB_database->Quote( $this->getString( 'title' ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'title' ) . " = " . $_CB_database->Quote( $this->getString( 'title' ) );
				}
			}

			if ( $this->getString( 'message', '' ) !== '' ) {
				if ( strpos( $this->getString( 'message' ), '%' ) !== false ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( $this->getString( 'message' ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " = " . $_CB_database->Quote( $this->getString( 'message' ) );
				}
			}

			if ( $this->getString( 'search', '' ) !== '' ) {
				$where[]					=	"( a." . $_CB_database->NameQuote( 'title' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false )
											.	" OR a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false )
											.	" OR a." . $_CB_database->NameQuote( 'date' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false ) . " )";
			}

			if ( $myId ) {
				if ( $this->getBool( 'hidden', false ) ) {
					$hiddenWhere			=	array();

					$select[]				=	"h1." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_id";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" ON h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' );

					$hiddenWhere[]			=	"h1." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$select[]				=	"h2." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_user";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h2"
											.	" ON h2." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h2." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity.user' )
											.	" AND h2." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'user_id' );

					$hiddenWhere[]			=	"h2." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$select[]				=	"h3." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_asset";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h3"
											.	" ON h3." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h3." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity.asset' )
											.	" AND h3." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' );

					$hiddenWhere[]			=	"h3." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$where[]				=	"( " . implode( " OR ", $hiddenWhere ) . " )";
				} else {
					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" WHERE h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' )
											.	" )";

					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h2"
											.	" WHERE h2." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h2." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity.user' )
											.	" AND h2." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	" )";

					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h3"
											.	" WHERE h3." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h3." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'activity.asset' )
											.	" AND h3." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' )
											.	" )";
				}
			}
		}

		if ( ( $this->getRaw( 'published' ) !== null ) && ( $this->getRaw( 'published' ) !== '' ) ) {
			$where[]						=	"a." . $_CB_database->NameQuote( 'published' ) . " = " . $this->getInt( 'published' );
		}

		$_PLUGINS->trigger( 'activity_onQueryActivityStream', array( $output, &$select, &$join, &$where, &$this ) );

		$query								=	"SELECT " . implode( ", ", $select )
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity' ) . " AS a"
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
											.	" ON cb." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
											.	" ON j." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	( $join ? "\n " . implode( "\n ", $join ) : null )
											.	"\n WHERE cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
											.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	( $where ? "\n AND " . implode( "\n AND ", $where ) : null );

		if ( ( ! $hasId ) && ( $output !== 'count' ) ) {
			$query							.=	"\n ORDER BY ";

			if ( $this->getBool( 'pinned', true ) ) {
				$query						.=	"a." . $_CB_database->NameQuote( 'pinned' ) . " DESC, ";
			}

			$query							.=	"a." . $_CB_database->NameQuote( 'date' ) . " DESC";
		}

		$cacheId							=	md5( $query . ( $output === 'count' ? $output : ( $output ? $output : null ) . ( $paging ? $this->getInt( 'paging_limitstart', 0 ) . $pageLimit : null ) ) );

		if ( ( ! isset( $cache[$cacheId] ) ) || ( ( ( $output === 'count' ) && $this->clearRowCount ) || $this->clearRowSelect ) ) {
			if ( $output === 'count' ) {
				$this->clearRowCount		=	false;

				$_CB_database->setQuery( $query );

				$cache[$cacheId]			=	(int) $_CB_database->loadResult();

				$this->set( 'query_count', $cache[$cacheId] );
			} else {
				$this->clearRowSelect		=	false;

				if ( $paging ) {
					$_CB_database->setQuery( $query, $this->getInt( 'paging_limitstart', 0 ), ( $pageLimit + 1 ) );
				} else {
					$_CB_database->setQuery( $query );
				}

				$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + $pageLimit ) );

				$rows						=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\ActivityTable', array( $_CB_database ) );
				$rowsCount					=	count( $rows );

				if ( $paging ) {
					$this->set( 'paging_total', $rowsCount );

					$rows					=	array_slice( $rows, 0, $pageLimit, true );
					$rowsCount				=	count( $rows );
				}

				$userIds					=	array();

				/** @var ActivityTable[] $rows */
				foreach ( $rows as $row ) {
					if ( preg_match( '/^profile\.(\d+)/', $row->getString( 'asset' ), $matches ) ) {
						$userIds[]			=	(int) $matches[1];
					}

					$userIds[]				=	$row->getInt( 'user_id', 0 );
				}

				if ( $userIds ) {
					\CBuser::advanceNoticeOfUsersNeeded( $userIds );
				}

				$_PLUGINS->trigger( 'activity_onLoadActivityStream', array( &$rows, $this ) );

				if ( $rows ) {
					static $group			=	array();
					static $parents			=	array();

					$streamId				=	$this->id();

					// Check if any rows have been told to group up by asset:
					foreach ( $rows as $k => $row ) {
						$asset								=	$row->params()->getString( 'overrides.group' );

						if ( $asset === 'asset' ) {
							$asset							=	$row->getString( 'asset' );
						} elseif ( ! $asset ) {
							$asset							=	$row->params()->getString( 'asset' );
						}

						if ( ! $asset ) {
							if ( $parents ) {
								// Check if this entries asset is a parent of a group:
								$asset						=	$row->getString( 'asset' );

								if ( isset( $parents[$asset] ) ) {
									unset( $rows[$k] );
								}

								// Check if this entries activity is a parent of a group:
								$asset						=	'activity.' . $row->getInt( 'id', 0 );

								if ( isset( $parents[$asset] ) ) {
									unset( $rows[$k] );
								}
							}

							continue;
						}

						$parent								=	$row->params()->getString( 'overrides.group_parent' );

						if ( $parent ) {
							$parents[$parent]				=	$parent;
						}

						$days								=	$row->params()->getInt( 'overrides.group_days' );

						if ( ! $days ) {
							$days							=	10;
						}

						if ( ! isset( $group[$streamId][$asset] ) ) {
							$group[$streamId][$asset]		=	$row;
						} else {
							$dateDiff						=	Application::Date( $group[$streamId][$asset]->getString( 'date' ), 'UTC' )->diff( $row->getString( 'date' ) );

							if ( $dateDiff->days <= $days ) {
								$names						=	$group[$streamId][$asset]->params()->getRaw( 'overrides.names', array() );

								if ( $group[$streamId][$asset]->getInt( 'user_id', 0 ) !== $row->getInt( 'user_id', 0 ) ) {
									$names[]				=	$row->getInt( 'user_id', 0 );
								}

								$group[$streamId][$asset]->params()->set( 'overrides.names', array_unique( $names ) );

								unset( $rows[$k] );
							} else {
								$group[$streamId][$asset]	=	$row;
							}
						}
					}

					if ( $rows ) {
						self::$loadedRows	+=	$rows;
					}
				}

				$newCount					=	count( $rows );

				if ( $paging && $rowsCount && ( $newCount < round( $rowsCount / 1.25 ) ) ) {
					$pagingTotal			=	$this->getInt( 'paging_total', 0 );
					$nextLimit				=	( $pageLimit - $newCount );

					if ( $nextLimit <= 0 ) {
						$nextLimit			=	1;
					}

					$this->set( 'paging_limit', $nextLimit );

					$cache[$cacheId]		=	( $rows + $this->rows( $output ) );

					$this->set( 'paging_total', $pagingTotal );
					$this->set( 'paging_limit', $pageLimit );
				} else {
					$cache[$cacheId]		=	$rows;
				}

				if ( $cache[$cacheId] ) {
					if ( $this->getBool( 'comments', true ) ) {
						CBActivity::prefetchAssets( 'comments', $cache[$cacheId], $this );
					}

					if ( $this->getBool( 'tags', true ) ) {
						CBActivity::prefetchAssets( 'tags', $cache[$cacheId], $this );
					}

					if ( $this->getBool( 'likes', true ) ) {
						CBActivity::prefetchAssets( 'likes', $cache[$cacheId], $this );
					}
				}
			}
		} elseif ( $output !== 'count' ) {
			$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + count( $cache[$cacheId] ) ) );
		}

		return $cache[$cacheId];
	}

	/**
	 * Retrieves activity row
	 *
	 * @param int $id
	 * @return ActivityTable
	 */
	public function row( $id )
	{
		if ( ! $id ) {
			return new ActivityTable();
		}

		if ( isset( self::$loadedRows[$id] ) ) {
			return self::$loadedRows[$id];
		}

		static $cache		=	array();

		if ( ! isset( $cache[$id] ) ) {
			$rows			=	$this->reset()->setId( $id )->rows();

			if ( isset( $rows[$id] ) ) {
				$row		=	$rows[$id];
			} else {
				$row		=	new ActivityTable();
			}

			$cache[$id]		=	$row;
		}

		return $cache[$id];
	}

	/**
	 * Outputs activity HTML
	 *
	 * @param null|string $view
	 * @param int         $id
	 * @param array       $params
	 * @return string
	 */
	public function activity( $view = null, $id = 0, $params = array() )
	{
		return $this->display( $view, $id, $params );
	}
}