<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\AntiSpam\Captcha;
use CB\Plugin\AntiSpam\CBAntiSpam;
use CB\Plugin\AntiSpam\Table\BlockTable;
use CB\Plugin\AntiSpam\Table\WhitelistTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined( 'CBLIB' ) or die();

class AntispamAction extends Action
{

	/**
	 * @param UserTable $user
	 * @return null|string|bool
	 */
	public function execute( UserTable $user )
	{
		global $_PLUGINS;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ANTISPAM_NOT_INSTALLED', ':: Action [action] :: CB AntiSpam is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return null;
		}

		$return							=	null;

		foreach ( $this->autoaction()->getParams()->subTree( 'antispam' ) as $row ) {
			/** @var ParamsInterface $row */
			$mode						=	$row->getString( 'mode', 'block' );

			if ( $mode === 'captcha' ) {
				$method					=	$row->getString( 'method', 'display' );
				$name					=	$this->string( $user, $row->getString( 'name', '' ) );

				if ( ! $name ) {
					$name				=	'autoaction' . $this->autoaction()->getId();
				}

				$error					=	$this->string( $user, $row->getString( 'error', '' ) );

				if ( ! $error ) {
					$error				=	CBTxt::T( 'Invalid Captcha Code' );
				}

				$captcha				=	new Captcha( $name, $row->getString( 'captcha', '-1' ) );

				if ( $method === 'validate' ) {
					if ( ( ! $captcha->load() ) || ( ! $captcha->validate() ) ) {
						$error			=	( $captcha->error() ?: $error );

						foreach ( $this->variables() as $variable ) {
							if ( ! \is_object( $variable ) ) {
								continue;
							}

							if ( method_exists( $variable, 'setError' ) ) {
								$variable->setError( $error );
							}
						}

						$_PLUGINS->_setErrorMSG( $error );
						$_PLUGINS->raiseError();

						$return			=	false;
					} else {
						$return			=	true;
					}
				} else {
					$return				=	$captcha->captcha()
										.	$captcha->input();
				}
			} else {
				$type					=	$row->getString( 'type', 'user' );
				$value					=	$row->getString( 'value', '' );

				if ( ! $value ) {
					switch ( $type ) {
						case 'account':
						case 'user':
							$value		=	$user->getInt( 'id', 0 );
							break;
						case 'ip':
							$value		=	CBAntiSpam::getUserIP( $user );
							break;
						case 'ip_range':
							$value		=	CBAntiSpam::getUserIP( $user ) . ':' . CBAntiSpam::getUserIP( $user );
							break;
						case 'email':
							$value		=	$user->getString( 'email', '' );
							break;
						case 'domain':
							$value		=	CBAntiSpam::getEmailDomain( $user );
							break;
					}
				} else {
					$value				=	$this->string( $user, $value );
				}

				if ( ! $value ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_ANTISPAM_NO_VALUE', ':: Action [action] :: CB AntiSpam skipped due to missing value', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				if ( $mode === 'block' ) {
					$duration			=	$row->getString( 'duration', '+1 MONTH' );

					if ( $duration === 'custom' ) {
						$duration		=	$this->string( $user, $row->getString( 'custom_duration', '' ) );
					}

					$entry				=	new BlockTable();

					$entry->set( 'date', Application::Database()->getUtcDateTime() );
					$entry->set( 'duration', $duration );
				} else {
					$entry				=	new WhitelistTable();
				}

				$entry->set( 'type', $type );
				$entry->set( 'value', $value );
				$entry->set( 'reason', $this->string( $user, $row->getString( 'reason', '' ) ) );

				if ( ! $entry->store() ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_ANTISPAM_FAILED', ':: Action [action] :: CB AntiSpam failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $entry->getError() ] ) );
				}
			}
		}

		return $return;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_PLUGINS;

		$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbantispam' );

		if ( ! $plugin ) {
			return false;
		}

		$pluginVersion		=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

		return version_compare( $pluginVersion, '3.0.0', '>=' ) && version_compare( $pluginVersion, '4.0.0', '<' );
	}
}