<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Table;

use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

class BatchTable extends Table
{
	/** @var null|int */
	public $id;
	/** @var null|string */
	public $title;
	/** @var null|int */
	public $users;
	/** @var null|string */
	public $access;
	/** @var null|string */
	public $actions;
	/** @var null|string */
	public $filter_basic;
	/** @var null|string */
	public $filter_join;
	/** @var null|string */
	public $filter_where;
	/** @var null|int */
	public $batch_limit;
	/** @var null|int */
	public $batch_progress;
	/** @var null|int */
	public $batch_redirect;
	/** @var null|int */
	public $use_limit;
	/** @var null|int */
	public $use_progress;
	/** @var null|int */
	public $published;

	/** @var null|Registry  */
	protected ?Registry $_filter_basic	=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_autoactions_batches';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check(): bool
	{
		if ( ! $this->getTitle() ) {
			$this->setError( CBTxt::T( 'Title not specified!' ) );

			return false;
		}

		if ( ! $this->getActions() ) {
			$this->setError( CBTxt::T( 'Auto Actions not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @return AutoActionTable[]
	 */
	public function getAutoActions(): array
	{
		global $_CB_database;

		static $cache			=	[];

		$actions				=	$this->getActions();

		if ( ! $actions ) {
			return [];
		}

		$id						=	$this->getId();

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]			=	[];

			$actions			=	new Registry( $actions );
			$actionIds			=	[];

			foreach ( $actions as $action ) {
				/** @var ParamsInterface $action */
				$actionId		=	$action->getInt( 'action', 0 );

				if ( ! $actionId ) {
					continue;
				}

				$actionIds[]	=	$actionId;
			}

			if ( ! $actionIds ) {
				return [];
			}

			$query				=	'SELECT *'
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
								.	"\n WHERE " . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $actionIds )
								.	"\n AND " . $_CB_database->NameQuote( 'published' ) . " = 1";
			$_CB_database->setQuery( $query );
			$autoActions		=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\AutoActions\Table\AutoActionTable', [ $_CB_database ] );

			foreach ( $actionIds as $actionId ) {
				if ( ! isset( $autoActions[$actionId] ) ) {
					continue;
				}

				$cache[$id][]	=	$autoActions[$actionId];
			}
		}

		return $cache[$id];
	}

	/**
	 * @return int
	 */
	public function getId(): int
	{
		return $this->getInt( 'id', 0 );
	}

	/**
	 * @param null|int $id
	 */
	public function setId( ?int $id ): void
	{
		$this->id	=	$id;
	}

	/**
	 * @return string
	 */
	public function getTitle(): string
	{
		return $this->getString( 'title', '' );
	}

	/**
	 * @param string $title
	 */
	public function setTitle( string $title ): void
	{
		$this->title	=	$title;
	}

	/**
	 * @return int
	 */
	public function getUsers(): int
	{
		return $this->getInt( 'users', 0 );
	}

	/**
	 * @param int $users
	 */
	public function setUsers( int $users ): void
	{
		$this->users	=	$users;
	}

	/**
	 * @return string
	 */
	public function getAccess(): string
	{
		return $this->getString( 'access', '-1' );
	}

	/**
	 * @param null|string $access
	 */
	public function setAccess( ?string $access ): void
	{
		$this->access	=	$access;
	}

	/**
	 * @return string
	 */
	public function getActions(): string
	{
		return $this->getRaw( 'actions', '' );
	}

	/**
	 * @param string $actions
	 */
	public function setActions( string $actions ): void
	{
		$this->actions	=	$actions;
	}

	/**
	 * @return Registry
	 */
	public function getFilterBasic(): Registry
	{
		if ( $this->_filter_basic === null ) {
			$this->_filter_basic	=	new Registry( $this->filter_basic );
		}

		return $this->_filter_basic;
	}

	/**
	 * @param string|array $filter_basic
	 */
	public function setFilterBasic( $filter_basic ): void
	{
		$this->getFilterBasic()->load( $filter_basic );
	}

	/**
	 * @return string
	 */
	public function getFilterJoin(): string
	{
		return $this->getRaw( 'filter_join', '' );
	}

	/**
	 * @param null|string $filter_join
	 */
	public function setFilterJoin( ?string $filter_join ): void
	{
		$this->filter_join	=	$filter_join;
	}

	/**
	 * @return string
	 */
	public function getFilterWhere(): string
	{
		return $this->getRaw( 'filter_where', '' );
	}

	/**
	 * @param null|string $filter_where
	 */
	public function setFilterWhere( ?string $filter_where ): void
	{
		$this->filter_where	=	$filter_where;
	}

	/**
	 * @return int
	 */
	public function getBatchLimit(): int
	{
		return $this->getInt( 'batch_limit', 0 );
	}

	/**
	 * @param int $batch_limit
	 */
	public function setBatchLimit( int $batch_limit ): void
	{
		$this->batch_limit	=	$batch_limit;
	}

	/**
	 * @return int
	 */
	public function getBatchProgress(): int
	{
		return $this->getInt( 'batch_progress', 0 );
	}

	/**
	 * Sets batch progress to the specified amount
	 * Returns true if successful and false if failed
	 *
	 * @param int $progress
	 * @return bool
	 */
	public function setBatchProgress( int $progress ): bool
	{
		if ( ! $this->getBatchLimit() ) {
			return true;
		}

		$this->set( 'batch_progress', $progress );

		return ( ( ! $this->getError() ) && $this->check() && $this->store() );
	}

	/**
	 * Increments the batch progress by the specified amount
	 * Returns true if successful and false if failed
	 *
	 * @param int $progress
	 * @return bool
	 */
	public function incrementBatchProgress( int $progress ): bool
	{
		if ( ! $this->getBatchLimit() ) {
			return true;
		}

		$this->set( 'batch_progress', ( $this->getBatchProgress() + $progress ) );

		return ( ( ! $this->getError() ) && $this->check() && $this->store() );
	}

	/**
	 * @return bool
	 */
	public function getBatchRedirect(): bool
	{
		return $this->getBool( 'batch_redirect', true );
	}

	/**
	 * @param bool $batch_redirect
	 */
	public function setBatchRedirect( bool $batch_redirect ): void
	{
		$this->batch_redirect	=	(int) $batch_redirect;
	}

	/**
	 * @return int
	 */
	public function getUseLimit(): int
	{
		return $this->getInt( 'use_limit', 0 );
	}

	/**
	 * @param int $use_limit
	 */
	public function setUseLimit( int $use_limit ): void
	{
		$this->use_limit	=	$use_limit;
	}

	/**
	 * @return int
	 */
	public function getUseProgress(): int
	{
		return $this->getInt( 'use_progress', 0 );
	}

	/**
	 * Sets batch use to the specified amount
	 * Returns true if successful and false if failed
	 *
	 * @param int $progress
	 * @return bool
	 */
	public function setUseProgress( int $progress ): bool
	{
		if ( ! $this->getUseLimit() ) {
			return true;
		}

		$this->set( 'use_progress', $progress );

		return ( ( ! $this->getError() ) && $this->check() && $this->store() );
	}

	/**
	 * Increments the use progress by the specified amount
	 * Returns true if successful and false if failed
	 *
	 * @param int $progress
	 * @return bool
	 */
	public function incrementUseProgress( int $progress ): bool
	{
		if ( ! $this->getUseLimit() ) {
			// Still reset the batch progress since we've completed a use:
			$this->setBatchProgress( 0 );

			return true;
		}

		if ( $this->getBatchLimit() ) {
			// Reset batch progress since we've completed a full batch process:
			$this->set( 'batch_progress', 0 );
		}

		$this->set( 'use_progress', ( $this->getUsers() + $progress ) );

		return ( ( ! $this->getError() ) && $this->check() && $this->store() );
	}

	/**
	 * @return int
	 */
	public function getPublished(): int
	{
		return $this->getInt( 'published', 0 );
	}

	/**
	 * @param int $published
	 */
	public function setPublished( int $published ): void
	{
		$this->published	=	$published;
	}
}