<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJiveForums\Forum\Kunena;

use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;
use CB\Plugin\GroupJive\CBGroupJive;
use CB\Plugin\GroupJive\Table\GroupTable;
use CB\Plugin\GroupJiveForums\Forum\ForumInterface;
use CB\Plugin\GroupJiveForums\Table\Kunena\CategoryTable;
use CB\Plugin\GroupJiveForums\Table\Kunena\PostTable;
use CB\Plugin\GroupJiveForums\CBGroupJiveForums;
use Kunena\Forum\Libraries\Forum\Category\KunenaCategoryHelper;
use Kunena\Forum\Libraries\Forum\KunenaStatistics;
use Kunena\Forum\Libraries\Forum\Message\KunenaMessage;
use Kunena\Forum\Libraries\Forum\Message\KunenaMessageHelper;

defined('CBLIB') or die();

class KunenaForum implements ForumInterface
{
	/** @var string  */
	public $type		=	'kunena';
	/** @var int  */
	public $version		=	0;

	/**
	 * @param int $version
	 */
	public function __construct( $version = 0 )
	{
		$this->version	=	$version;
	}

	/**
	 * Returns an a cached count of the number of available categories
	 *
	 * @return int
	 */
	public function getCategoriesCount()
	{
		if ( $this->version >= 6 ) {
			$statistics		=	KunenaStatistics::getInstance();
		} else {
			$statistics		=	\KunenaForumStatistics::getInstance();
		}

		$statistics->loadCategoryCount();

		return (int) $statistics->categoryCount;
	}

	/**
	 * Returns an array of available categories
	 *
	 * @return array
	 */
	public function getCategories()
	{
		if ( $this->version >= 6 ) {
			$rows		=	KunenaCategoryHelper::getChildren( 0, 10, array( 'action' => 'admin', 'unpublished' => true ) );
		} else {
			$rows		=	\KunenaForumCategoryHelper::getChildren( 0, 10, array( 'action' => 'admin', 'unpublished' => true ) );
		}

		$options		=	array();

		foreach ( $rows as $row ) {
			$options[]	=	\moscomprofilerHTML::makeOption( (string) $row->id, str_repeat( '- ', $row->level + 1  ) . ' ' . $row->name );
		}

		return $options;
	}

	/**
	 * Returns a forum category object
	 *
	 * @param int $id
	 * @return CategoryTable
	 */
	public function getCategory( $id )
	{
		if ( ! $id ) {
			return new CategoryTable();
		}

		static $cache		=	array();

		if ( ! isset( $cache[$id] ) ) {
			$row			=	new CategoryTable();

			$row->load( (int) $id );

			$cache[$id]		=	$row;
		}

		return $cache[$id];
	}

	/**
	 * Returns a display array of kunena topics for a group
	 *
	 * @param UserTable  $user
	 * @param GroupTable $group
	 * @param array      $counters
	 * @return array|null
	 */
	public function getTopics( $user, &$group, &$counters )
	{
		global $_CB_framework, $_CB_database;

		$categoryId					=	$group->params()->getInt( 'forum_id', 0 );

		if ( ( ! $categoryId ) || ( ! $group->params()->getBool( 'forums', true ) ) || ( $group->category()->getInt( 'id', 0 ) && ( ! $group->category()->params()->getBool( 'forums', true ) ) ) ) {
			return null;
		}

		CBGroupJive::getTemplate( 'forums', true, true, 'cbgroupjiveforums' );

		$prefix						=	'gj_group_' . $group->getInt( 'id', 0 ) . '_forums_';
		$limit						=	CBGroupJiveForums::getGlobalParams()->getInt( 'groups_forums_limit', 15 );
		$limitstart					=	$_CB_framework->getUserStateFromRequest( $prefix . 'limitstart{com_comprofiler}', $prefix . 'limitstart' );
		$search						=	$_CB_framework->getUserStateFromRequest( $prefix . 'search{com_comprofiler}', $prefix . 'search' );
		$where						=	null;

		if ( $search && CBGroupJiveForums::getGlobalParams()->getBool( 'groups_forums_search', true ) ) {
			$where					.=	'( m.' . $_CB_database->NameQuote( 'subject' ) . ' LIKE ' . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
									.	' OR t.' . $_CB_database->NameQuote( 'message' ) . ' LIKE ' . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . ' )';
		}

		$searching					=	( $where ? true : false );

		$params						=	array(	'starttime'	=>	-1,
												'where'		=>	$where
											);

		if ( $this->version >= 6 ) {
			$posts					=	KunenaMessageHelper::getLatestMessages( $categoryId, 0, 0, $params );
		} else {
			$posts					=	\KunenaForumMessageHelper::getLatestMessages( $categoryId, 0, 0, $params );
		}

		$total						=	array_shift( $posts );

		if ( ( ! $total ) && ( ! $searching ) && ( ! CBGroupJive::canCreateGroupContent( $user, $group, 'forums' ) ) ) {
			return null;
		}

		$pageNav					=	new \cbPageNav( $total, $limitstart, $limit );

		$pageNav->setInputNamePrefix( $prefix );
		$pageNav->setStaticLimit( true );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( 'cbgroupjive', false, array( 'action' => 'groups', 'func' => 'show', 'id' => $group->getInt( 'id', 0 ), $prefix . 'search' => ( $searching ? $search : null ) ) ) );

		if ( CBGroupJiveForums::getGlobalParams()->getInt( 'groups_forums_orderby', 2 ) === 1 ) {
			$params['orderby']		=	'm.' . $_CB_database->NameQuote( 'time' ) . ' ASC';
		}

		if ( CBGroupJiveForums::getGlobalParams()->getBool( 'groups_forums_paging', true ) ) {
			if ( $this->version >= 6 ) {
				$posts				=	KunenaMessageHelper::getLatestMessages( $categoryId, (int) $pageNav->limitstart, (int) $pageNav->limit, $params );
			} else {
				$posts				=	\KunenaForumMessageHelper::getLatestMessages( $categoryId, (int) $pageNav->limitstart, (int) $pageNav->limit, $params );
			}

			$posts					=	array_pop( $posts );
		} else {
			$posts					=	array_pop( $posts );
		}

		$rows						=	array();

		/** @var KunenaMessage[]|\KunenaForumMessage[] $posts */
		foreach ( $posts as $post ) {
			$row					=	new PostTable();

			$row->post( $post );

			$rows[]					=	$row;
		}

		$input						=	array();

		$input['search']			=	'<input type="text" name="' . htmlspecialchars( $prefix ) . 'search" value="' . htmlspecialchars( $search ) . '" placeholder="' . htmlspecialchars( CBTxt::T( 'Search Posts...' ) ) . '" class="form-control" />';

		CBGroupJive::prefetchUsers( $rows );

		$group->set( '_forums', $pageNav->total );

		return array(	'id'		=>	'forums',
						'title'		=>	CBTxt::T( 'Forums' ),
						'content'	=>	\HTML_groupjiveForums::showForums( $rows, $pageNav, $searching, $input, $counters, $group, $user, $this )
					);
	}
}