<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions;

use CB\Plugin\AutoActions\Table\AutoActionTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CBLib\Xml\SimpleXMLElement;

\defined( 'CBLIB' ) or die();

class SystemActions
{

	/**
	 * Installs system actions
	 *
	 * @return string
	 */
	public static function install(): string
	{
		global $_CB_framework, $_CB_database, $_PLUGINS;

		$_PLUGINS->loadPluginGroup( null, null, 0 );

		$loadedPlugins							=	$_PLUGINS->getLoadedPluginGroup( null );
		$order									=	1;
		$exist									=	[];
		$autoactions							=	[];
		$return									=	null;

		foreach ( $loadedPlugins as $plugin ) {
			$element							=	$_PLUGINS->loadPluginXML( 'autoactions', 'system', $plugin->id );

			$pluginAutoActions					=	$element->xpath( '/autoactions/autoaction' );

			if ( $pluginAutoActions && \count( $pluginAutoActions ) ) {
				$autoactions[$plugin->element]	=	$pluginAutoActions;
			}
		}

		// CB Auto Actions couldn't be loaded yet (typically during install) so lets force load the core system actions:
		if ( ! isset( $autoactions['cbautoactions'] ) ) {
			$xml									=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbautoactions/xml/edit.autoactions.system.xml';

			if ( file_exists( $xml ) ) {
				$element							=	new SimpleXMLElement( trim( file_get_contents( $xml ) ) );

				$coreAutoActions					=	$element->xpath( '/autoactions/autoaction' );

				if ( $coreAutoActions && \count( $coreAutoActions ) ) {
					$autoactions['cbautoactions']	=	$coreAutoActions;
				}
			}
		}

		foreach ( $autoactions as $plugin => $pluginAutoActions ) {
			foreach ( $pluginAutoActions as $autoaction ) {
				/** @var SimpleXMLElement $autoaction */
				$name							=	(string) $autoaction->attributes( 'name' );
				$type							=	(string) $autoaction->attributes( 'type' );

				if ( ( ! $name ) || ( ! $type ) ) {
					continue;
				}

				$label							=	(string) $autoaction->attributes( 'label' );

				if ( ! $label ) {
					$label						=	$name;
				}

				$action							=	new AutoActionTable( $_CB_database, '#__comprofiler_plugin_autoactions', 'id' );

				$action->load( [ 'system' => (string) $autoaction->attributes( 'name' ) ] );

				if ( ! $action->getInt( 'id', 0 ) ) {
					$action->set( 'published', (int) $autoaction->attributes( 'published' ) );
				}

				$action->set( 'system', $name );
				$action->set( 'title', $label );
				$action->set( 'type', $type );
				$action->set( 'ordering', $order );

				$variables						=	$autoaction->xpath( 'param' );

				if ( $variables && \count( $variables ) ) {
					foreach ( $variables as $variable ) {
						$varName				=	(string) $variable->attributes( 'name' );

						if ( ! $varName ) {
							continue;
						}

						if ( $variable->attributes( 'value' ) === null ) {
							$action->set( $varName, trim( (string) $variable ) );
						} else {
							$action->set( $varName, (string) $variable->attributes( 'value' ) );
						}
					}
				}

				if ( $action->getString( 'object' ) === null ) {
					$action->set( 'object', 0 );
				}

				if ( $action->getString( 'variable' ) === null ) {
					$action->set( 'variable', 1 );
				}

				if ( $action->getString( 'access' ) === null ) {
					$action->set( 'access', -1 );
				}

				$actionConditions						=	new Registry();
				/** @var SimpleXMLElement[] $orConditions */
				$orConditions							=	$autoaction->xpath( 'conditions/or' );

				if ( $orConditions && \count( $orConditions ) ) {
					foreach ( $orConditions as $orIndex => $orCondition ) {
						$andConditions					=	$orCondition->xpath( 'and' );

						if ( $andConditions && \count( $andConditions ) ) {
							foreach ( $andConditions as $andIndex => $andCondition ) {
								$params					=	$andCondition->xpath( 'param' );

								if ( $params && \count( $params ) ) {
									foreach ( $params as $param ) {
										$paramName		=	(string) $param->attributes( 'name' );

										if ( ! $paramName ) {
											continue;
										}

										$options		=	[];

										if ( $param->attributes( 'translate' ) === 'true' ) {
											$options[]	=	'translate';
										}

										if ( $param->attributes( 'format' ) === 'true' ) {
											$options[]	=	'format';
										}

										if ( $param->attributes( 'content_plugins' ) === 'true' ) {
											$options[]	=	'content_plugins';
										}

										if ( $paramName === 'field_custom' ) {
											$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.field', 'customvalue' );

											if ( $options ) {
												$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.field_custom_options', implode( '|*|', $options ) );
											}
										} elseif ( $paramName === 'field_code' ) {
											$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.field', 'customcode' );

											if ( $options ) {
												$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.field_code_options', implode( '|*|', $options ) );
											}
										} elseif ( $paramName === 'value' ) {
											if ( $options ) {
												$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.value_options', implode( '|*|', $options ) );
											}
										}

										if ( $param->attributes( 'value' ) === null ) {
											$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.' . $paramName, trim( (string) $param ) );
										} else {
											$actionConditions->set( $orIndex . '.condition.' . $andIndex . '.' . $paramName, (string) $param->attributes( 'value' ) );
										}
									}
								}
							}
						}
					}
				}

				$action->set( 'conditions', $actionConditions->asJson() );

				$actionParams					=	new Registry();
				$repeatActions					=	$autoaction->xpath( 'actions/action' );

				if ( $repeatActions && \count( $repeatActions ) ) {
					$actionsName				=	(string) $autoaction->xpath( 'actions' )[0]->attributes( 'name' );

					if ( ! $actionsName ) {
						$actionsName			=	$type;
					}

					foreach ( $repeatActions as $i => $repeatAction ) {
						$params					=	$repeatAction->xpath( 'param' );

						if ( $params && \count( $params ) ) {
							foreach ( $params as $param ) {
								$paramName		=	(string) $param->attributes( 'name' );

								if ( ! $paramName ) {
									continue;
								}

								if ( $param->attributes( 'value' ) === null ) {
									$actionParams->set( $actionsName . '.' . $i . '.' . $paramName, trim( (string) $param ) );
								} else {
									$actionParams->set( $actionsName . '.' . $i . '.' . $paramName, (string) $param->attributes( 'value' ) );
								}
							}
						}
					}
				}

				$singleAction					=	$autoaction->xpath( 'action/param' );

				if ( $singleAction && \count( $singleAction ) ) {
					$actionsName				=	(string) $autoaction->xpath( 'action' )[0]->attributes( 'name' );

					if ( ! $actionsName ) {
						$actionsName			=	$type;
					}

					foreach ( $singleAction as $param ) {
						$paramName				=	(string) $param->attributes( 'name' );

						if ( ! $paramName ) {
							continue;
						}

						if ( $param->attributes( 'value' ) === null ) {
							$actionParams->set( $actionsName . '.' . $paramName, trim( (string) $param ) );
						} else {
							$actionParams->set( $actionsName . '.' . $paramName, (string) $param->attributes( 'value' ) );
						}
					}
				}

				$params							=	$autoaction->xpath( 'params/param' );

				if ( $params && \count( $params ) ) {
					foreach ( $params as $param ) {
						$paramName				=	(string) $param->attributes( 'name' );

						if ( ! $paramName ) {
							continue;
						}

						if ( $param->attributes( 'value' ) === null ) {
							$actionParams->set( $paramName, trim( (string) $param ) );
						} else {
							$actionParams->set( $paramName, (string) $param->attributes( 'value' ) );
						}
					}
				}

				$action->set( 'params', $actionParams->asJson() );

				if ( $action->store() ) {
					$return						.=	'<p>'
												.		'<strong>' . $order . '. ' . $action->getString( 'title', '' ) . '</strong>'
												.		'<br />' . $action->getString( 'description', '' )
												.	'</p>';

					$order++;

					$exist[]					=	$action->getString( 'system', '' );
				}
			}
		}

		// Delete system actions that no longer exist:
		if ( $exist ) {
			$query								=	"DELETE"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'system' ) . " NOT IN " . $_CB_database->safeArrayOfStrings( $exist )
												.	"\n AND " . $_CB_database->NameQuote( 'system' ) . " != ''";
			$_CB_database->setQuery( $query );
			$_CB_database->query();

			$return								=	'<strong>' . CBTxt::T( 'COUNT_SYSTEM_ACTIONS_INSTALLED', '%%COUNT%% system action installed.|%%COUNT%% system actions installed.', [ '%%COUNT%%' => \count( $exist ) ] ) . '</strong><hr />'
												.	$return;
		} else {
			$return								=	CBTxt::T( 'No system actions installed!' );
		}

		return $return;
	}
}

/**
* System Action Language Strings
*
* CBTxt::T( 'Auto Login' )
* CBTxt::T( 'Automatically logs in a user after registration or confirmation (must be approved and confirmed).' )
* CBTxt::T( 'Profile - Logged In' )
* CBTxt::T( 'Logs activity when a user logs in.' )
* CBTxt::T( 'Profile - Logged Out' )
* CBTxt::T( 'Logs activity when a user logs out.' )
* CBTxt::T( 'Profile - Register' )
* CBTxt::T( 'Logs activity when a new user registers.' )
* CBTxt::T( 'Profile - Update' )
* CBTxt::T( 'Logs activity for profile updates.' )
* CBTxt::T( 'Profile - Avatar' )
* CBTxt::T( 'Logs activity for avatar updates.' )
* CBTxt::T( 'Profile - Canvas' )
* CBTxt::T( 'Logs activity for canvas updates.' )
* CBTxt::T( 'Page - Like' )
* CBTxt::T( 'Logs activity for liking a page.' )
* CBTxt::T( 'Page - Comment' )
* CBTxt::T( 'Logs activity for comments on pages.' )
* CBTxt::T( 'CB Connections - Add' )
* CBTxt::T( 'Logs activity when a new connection is added and accepted.' )
* CBTxt::T( 'CB Connections - Request Notification' )
* CBTxt::T( 'Logs notification when a new connection is requested and pending. Can be toggled with a checkbox field named cb_notif_request_conn.' )
* CBTxt::T( 'CB Connections - Accepted Notification' )
* CBTxt::T( 'Logs notification when a new connection is accepted. Can be toggled with a checkbox field named cb_notif_accepted_conn.' )
* CBTxt::T( 'CB Connections - Rejected Notification' )
* CBTxt::T( 'Logs notification when a new connection is rejected. Can be toggled with a checkbox field named cb_notif_rejected_conn.' )
* CBTxt::T( 'CB Connections - Remove' )
* CBTxt::T( 'Deletes activity and notifications for removed connections.' )
* CBTxt::T( 'CB Activity - Post Notification' )
* CBTxt::T( 'Logs notifications for activity posts on a users profile stream. Can be toggled with a checkbox field named cb_notif_post.' )
* CBTxt::T( 'CB Activity - Global Post Notification' )
* CBTxt::T( 'Logs notifications for global activity posts on any activity stream. Can be toggled with a checkbox field named cb_notif_global_post.' )
* CBTxt::T( 'CB Activity - Comment' )
* CBTxt::T( 'Logs activity for activity comments.' )
* CBTxt::T( 'CB Activity - Comment Bump' )
* CBTxt::T( 'Bumps activity to the top when a comment is made.' )
* CBTxt::T( 'CB Activity - Comment Notification' )
* CBTxt::T( 'Logs notifications for activity comments. Can be toggled with a checkbox field named cb_notif_comment.' )
* CBTxt::T( 'CB Activity - Comment Chain Notification' )
* CBTxt::T( 'Logs notifications for activity comments to all users (except activity owner) who commented. Can be toggled with a checkbox field named cb_notif_comment_chain.' )
* CBTxt::T( 'CB Activity - Comment Reply Notification' )
* CBTxt::T( 'Logs notifications for activity comment replies. Can be toggled with a checkbox field named cb_notif_comment_reply.' )
* CBTxt::T( 'CB Activity - Comment Reply Chain Notification' )
* CBTxt::T( 'Logs notifications for activity comment replies to all users (except comment owner) who commented. Can be toggled with a checkbox field named cb_notif_comment_reply_chain.' )
* CBTxt::T( 'CB Activity - Comment Tag' )
* CBTxt::T( 'Logs activity for comment tags.' )
* CBTxt::T( 'CB Activity - Comment Tag Notification' )
* CBTxt::T( 'Logs notifications for comment tags. Can be toggled with a checkbox field named cb_notif_comment_tag.' )
* CBTxt::T( 'CB Activity - Comment Like' )
* CBTxt::T( 'Logs activity for comment likes.' )
* CBTxt::T( 'CB Activity - Comment Like Notification' )
* CBTxt::T( 'Logs notifications for comment likes. Can be toggled with a checkbox field named cb_notif_comment_like.' )
* CBTxt::T( 'CB Activity - Activity Tag' )
* CBTxt::T( 'Logs activity for activity tags.' )
* CBTxt::T( 'CB Activity - Activity Tag Notification' )
* CBTxt::T( 'Logs notifications for activity tags. Can be toggled with a checkbox field named cb_notif_tag.' )
* CBTxt::T( 'CB Activity - Activity Like' )
* CBTxt::T( 'Logs activity for activity likes.' )
* CBTxt::T( 'CB Activity - Activity Like Notification' )
* CBTxt::T( 'Logs notifications for activity likes. Can be toggled with a checkbox field named cb_notif_like.' )
* CBTxt::T( 'CB Activity - Privacy - New' )
* CBTxt::T( 'Displays privacy selector for new activity.' )
* CBTxt::T( 'CB Activity - Privacy - Edit' )
* CBTxt::T( 'Displays privacy selector for activity edit.' )
* CBTxt::T( 'CB Activity - Privacy - Save' )
* CBTxt::T( 'Saves activity privacy rules.' )
* CBTxt::T( 'CB Activity - Privacy - Authorized' )
* CBTxt::T( 'Checks privacy rules on activity.' )
* CBTxt::T( 'CB Gallery - Create' )
* CBTxt::T( 'Logs activity for uploaded and linked media to profiles.' )
* CBTxt::T( 'CB Gallery - Delete' )
* CBTxt::T( 'Deletes activity, notifications, comments, tags and likes for deleted media.' )
* CBTxt::T( 'CB Gallery - Likes' )
* CBTxt::T( 'Adds likes to gallery media.' )
* CBTxt::T( 'CB Gallery - Comments' )
* CBTxt::T( 'Adds commenting to gallery media.' )
* CBTxt::T( 'CB Gallery - Create Connections Notification' )
* CBTxt::T( 'Logs notifications to users connections for uploaded and linked media. Can be toggled with a checkbox field named cb_notif_gallery_conn.' )
* CBTxt::T( 'CB Gallery - Comment Notification' )
* CBTxt::T( 'Logs notifications for gallery media comments. Can be toggled with a checkbox field named cb_notif_gallery_comment.' )
* CBTxt::T( 'CB Gallery - Tag Notification' )
* CBTxt::T( 'Logs notifications for gallery media tags. Can be toggled with a checkbox field named cb_notif_gallery_tag.' )
* CBTxt::T( 'CB Gallery - Like Notification' )
* CBTxt::T( 'Logs notifications for gallery media likes. Can be toggled with a checkbox field named cb_notif_gallery_like.' )
* CBTxt::T( 'CB Gallery - Privacy - Edit' )
* CBTxt::T( 'Displays privacy selector for gallery album and media edit.' )
* CBTxt::T( 'CB Gallery - Privacy - Save' )
* CBTxt::T( 'Saves gallery privacy rules.' )
* CBTxt::T( 'CB Gallery - Privacy - Authorized' )
* CBTxt::T( 'Checks privacy rules on gallery.' )
* CBTxt::T( 'CB Articles - Create' )
* CBTxt::T( 'Logs activity for newly created article entries.' )
* CBTxt::T( 'CB Articles - Comment' )
* CBTxt::T( 'Logs activity for comments on articles.' )
* CBTxt::T( 'CB Articles - Comment Notification' )
* CBTxt::T( 'Logs notifications for article comments. Can be toggled with a checkbox field named cb_notif_article_comment.' )
* CBTxt::T( 'CB Articles - Like Notification' )
* CBTxt::T( 'Logs notifications for article likes. Can be toggled with a checkbox field named cb_notif_article_like.' )
* CBTxt::T( 'CB Articles - Delete' )
* CBTxt::T( 'Deletes activity, notifications, comments and likes for deleted article entries.' )
* CBTxt::T( 'PMS - Create Notification' )
* CBTxt::T( 'Logs notification when a user receives a private message. Note this only applies to Internal model and does not function with UddeIM and only for user to user private messages. Can be toggled with a checkbox field named cb_notif_pm.' )
* CBTxt::T( 'PMS - Delete' )
* CBTxt::T( 'Deletes notifications for deleted private messages.' )
* CBTxt::T( 'CB Blogs - Create' )
* CBTxt::T( 'Logs activity for newly created blog entries.' )
* CBTxt::T( 'CB Blogs - Create Connections Notification' )
* CBTxt::T( 'Logs notifications to users connections for newly created blog entries. Can be toggled with a checkbox field named cb_notif_blog_conn.' )
* CBTxt::T( 'CB Blogs - Delete' )
* CBTxt::T( 'Deletes activity, notifications, comments and likes for deleted blog entries.' )
* CBTxt::T( 'Kunena - Create' )
* CBTxt::T( 'Logs activity for newly created discussions.' )
* CBTxt::T( 'Kunena - Reply' )
* CBTxt::T( 'Logs activity for discussion replies.' )
* CBTxt::T( 'Kunena - Delete' )
* CBTxt::T( 'Deletes activity for deleted discussions.' )
* CBTxt::T( 'CB GroupJive - Group Create' )
* CBTxt::T( 'Logs activity for newly created groups.' )
* CBTxt::T( 'CB GroupJive - Group Create Connections Notification' )
* CBTxt::T( 'Logs notifications to users connections for newly created groups. Can be toggled with a checkbox field named cb_notif_group_create_conn.' )
* CBTxt::T( 'CB GroupJive - Group Delete' )
* CBTxt::T( 'Deletes activity, notifications, follows, tags and likes for deleted groups.' )
* CBTxt::T( 'CB GroupJive - Group Join' )
* CBTxt::T( 'Logs activity when joining a group.' )
* CBTxt::T( 'CB GroupJive - Group Join Notification' )
* CBTxt::T( 'Logs notifications when joining a group. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive - Group Join Connections Notification' )
* CBTxt::T( 'Logs notifications to users connections when joining a group. Can be toggled with a checkbox field named cb_notif_group_join_conn.' )
* CBTxt::T( 'CB GroupJive - Group Leave' )
* CBTxt::T( 'Deletes group activity on group leave.' )
* CBTxt::T( 'CB GroupJive - Group Leave Notification' )
* CBTxt::T( 'Logs notifications when leaving a group. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive - Group Invite Notification' )
* CBTxt::T( 'Logs notifications when invited to a group. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive - Group Message Notification' )
* CBTxt::T( 'Logs notifications when sending group messages. This will override the GJ notification method.' )
* CBTxt::T( 'CB Groupjive - Stream' )
* CBTxt::T( 'Adds activity stream to groups.' )
* CBTxt::T( 'CB Groupjive - Stream Notification Edit' )
* CBTxt::T( 'Displays a notification parameter for toggling stream notifications on/off on a per group basis.' )
* CBTxt::T( 'CB Groupjive - Stream Notification' )
* CBTxt::T( 'Logs notifications to all active group users when a new post is made on the groups stream. Can be toggled with a checkbox field named cb_notif_group_stream.' )
* CBTxt::T( 'CB Groupjive - Follow' )
* CBTxt::T( 'Adds follow button to groups. Note only applies to Open groups.' )
* CBTxt::T( 'CB Groupjive - Likes' )
* CBTxt::T( 'Adds like button to groups.' )
* CBTxt::T( 'CB Groupjive - Gallery' )
* CBTxt::T( 'Adds media gallery to groups.' )
* CBTxt::T( 'CB Groupjive - Gallery Create' )
* CBTxt::T( 'Logs activity for uploaded and linked media in groups.' )
* CBTxt::T( 'CB Groupjive - Gallery Notification Edit' )
* CBTxt::T( 'Displays a notification parameter for toggling gallery notifications on/off on a per group basis.' )
* CBTxt::T( 'CB Groupjive - Gallery Notification' )
* CBTxt::T( 'Logs notifications to all active group users when media is uploaded or linked on a groups gallery. Can be toggled with a checkbox field named cb_notif_group_gallery.' )
* CBTxt::T( 'CB GroupJive - General Notifications' )
* CBTxt::T( 'Attempts to log all other notifications instead of sending private messages or emails.' )
* CBTxt::T( 'CB GroupJive Wall - Create' )
* CBTxt::T( 'Logs activity for newly created group wall posts.' )
* CBTxt::T( 'CB GroupJive Wall - Reply' )
* CBTxt::T( 'Logs comments for newly created group wall post replies.' )
* CBTxt::T( 'CB GroupJive Wall - Notifications' )
* CBTxt::T( 'Logs notifications for group wall posts. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive Wall - Delete' )
* CBTxt::T( 'Deletes activity, notifications and comments for deleted group wall posts.' )
* CBTxt::T( 'CB Groupjive Photo - Likes' )
* CBTxt::T( 'Adds like button to group photos.' )
* CBTxt::T( 'CB GroupJive Photo - Create' )
* CBTxt::T( 'Logs activity for newly uploaded group photos.' )
* CBTxt::T( 'CB GroupJive Photo - Notifications' )
* CBTxt::T( 'Logs notifications for group photos. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive Photo - Delete' )
* CBTxt::T( 'Deletes activity, notifications and likes for deleted group photos.' )
* CBTxt::T( 'CB GroupJive File - Create' )
* CBTxt::T( 'Logs activity for newly uploaded group files.' )
* CBTxt::T( 'CB GroupJive File - Notifications' )
* CBTxt::T( 'Logs notifications for group files. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive File - Delete' )
* CBTxt::T( 'Deletes activity, notifications and likes for deleted group files.' )
* CBTxt::T( 'CB Groupjive Video - Likes' )
* CBTxt::T( 'Adds like button to group videos.' )
* CBTxt::T( 'CB GroupJive Video - Create' )
* CBTxt::T( 'Logs activity for newly uploaded group videos.' )
* CBTxt::T( 'CB GroupJive Video - Notifications' )
* CBTxt::T( 'Logs notifications for group videos. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive Video - Delete' )
* CBTxt::T( 'Deletes activity, notifications and likes for deleted group videos.' )
* CBTxt::T( 'CB Groupjive Events - Likes' )
* CBTxt::T( 'Adds like button to group events.' )
* CBTxt::T( 'CB GroupJive Events - Create' )
* CBTxt::T( 'Logs activity for newly scheduled events.' )
* CBTxt::T( 'CB GroupJive Events - Notifications' )
* CBTxt::T( 'Logs notifications for group events. This will override the GJ notification method.' )
* CBTxt::T( 'CB GroupJive Events - Delete' )
* CBTxt::T( 'Deletes activity, notifications and likes for deleted group events.' )
*/