<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Table;

use CB\Database\Table\UserTable;
use CB\Plugin\AutoActions\Action\Action;
use CB\Plugin\AutoActions\Action\ActionInterface;
use CB\Plugin\AutoActions\CBAutoActions;
use CBLib\Application\Application;
use CBLib\Database\Table\OrderedTable;
use CBLib\Database\Table\TableInterface;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

class AutoActionTable extends OrderedTable
{
	/** @var null|int */
	public $id;
	/** @var null|string */
	public $system;
	/** @var null|string */
	public $title;
	/** @var null|string */
	public $description;
	/** @var null|string */
	public $type;
	/** @var null|string */
	public $trigger;
	/** @var null|string */
	public $object;
	/** @var null|string */
	public $variable;
	/** @var null|string */
	public $access;
	/** @var null|string */
	public $conditions;
	/** @var null|int */
	public $published;
	/** @var null|int */
	public $ordering;
	/** @var null|string */
	public $params;

	/** @var null|Registry  */
	protected ?Registry $_params		=	null;
	/** @var null|Registry  */
	protected ?Registry $_conditions	=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_autoactions';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check(): bool
	{
		if ( ! $this->getType() ) {
			$this->setError( CBTxt::T( 'Type not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false ): bool
	{
		if ( ! $this->getSystem() ) {
			$this->setSystem( '' );
		}

		return ( parent::store( $updateNulls ) !== false );
	}

	/**
	 * Generic check for whether dependencies exist for this object in the db schema
	 * Should be overridden if checks need to be done before delete()
	 *
	 * @param  int  $oid  key index (only int supported here)
	 * @return bool
	 */
	public function canDelete( $oid = null ): bool
	{
		if ( $this->getSystem() ) {
			$this->setError( CBTxt::T( 'System actions can not be deleted' ) );

			return false;
		}

		return true;
	}

	/**
	 * Copies this record (no checks)
	 * canCopy should be called first to check if a copy is possible.
	 *
	 * @param  null|TableInterface|self  $object  The object being copied otherwise create new object and add $this
	 * @return self|bool                       OBJECT: The new object copied successfully, FALSE: Failed to copy
	 */
	public function copy( $object = null )
	{
		if ( $object === null ) {
			$object		=	clone $this;
		}

		if ( $object->getSystem() ) {
			$object->setSystem( '' );
		}

		return parent::copy( $object );
	}

	/**
	 * @param null|UserTable $user
	 * @return int
	 */
	public function getUses( ?UserTable $user = null ): int
	{
		global $_CB_database;

		if ( $user ) {
			$user					=	$user->getInt( 'id', 0 );
		}

		static $cache				=	[];

		$id							=	$this->getId();

		if ( ! isset( $cache[$user][$id] ) ) {
			$query					=	"SELECT COUNT(*)"
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions_uses' );

			if ( $user ) {
				$timeframe			=	$this->getParams()->getString( 'uses_user_timeframe', 'forever' );
				$timeframeStart		=	$this->getParams()->getString( 'uses_user_timeframe_start', 'TODAY' );
				$timeframeEnd		=	$this->getParams()->getString( 'uses_user_timeframe_end', 'TOMORROW' );

				$query				.=	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $user
									.	"\n AND " . $_CB_database->NameQuote( 'auto_action' ) . " = " . $id;
			} else {
				$timeframe			=	$this->getParams()->getString( 'uses_total_timeframe', 'forever' );
				$timeframeStart		=	$this->getParams()->getString( 'uses_total_timeframe_start', 'TODAY' );
				$timeframeEnd		=	$this->getParams()->getString( 'uses_total_timeframe_end', 'TOMORROW' );

				$query				.=	"\n WHERE " . $_CB_database->NameQuote( 'auto_action' ) . " = " . $id;
			}

			if ( $timeframe !== 'forever' ) {
				switch( $timeframe ) {
					case 'daily':
						$query		.=	"\n AND " . $_CB_database->NameQuote( 'date' ) . " >= " . $_CB_database->Quote( Application::Date( 'TODAY', 'UTC' )->format( 'Y-m-d H:i:s' ) )
									.	"\n AND " . $_CB_database->NameQuote( 'date' ) . " < " . $_CB_database->Quote( Application::Date( 'TOMORROW', 'UTC' )->format( 'Y-m-d H:i:s' ) );
						break;
					case 'weekly':
						$query		.=	"\n AND " . $_CB_database->NameQuote( 'date' ) . " >= " . $_CB_database->Quote( Application::Date( 'MIDNIGHT MONDAY THIS WEEK', 'UTC' )->format( 'Y-m-d H:i:s' ) )
									.	"\n AND " . $_CB_database->NameQuote( 'date' ) . " < " . $_CB_database->Quote( Application::Date( 'MIDNIGHT MONDAY NEXT WEEK', 'UTC' )->format( 'Y-m-d H:i:s' ) );
						break;
					case 'monthly':
						$query		.=	"\n AND " . $_CB_database->NameQuote( 'date' ) . " >= " . $_CB_database->Quote( Application::Date( 'MIDNIGHT FIRST DAY OF THIS MONTH', 'UTC' )->format( 'Y-m-d H:i:s' ) )
									.	"\n AND " . $_CB_database->NameQuote( 'date' ) . " < " . $_CB_database->Quote( Application::Date( 'MIDNIGHT FIRST DAY OF NEXT MONTH', 'UTC' )->format( 'Y-m-d H:i:s' ) );
						break;
					case 'yearly':
						$query		.=	"\n AND " . $_CB_database->NameQuote( 'date' ) . " >= " . $_CB_database->Quote( Application::Date( 'MIDNIGHT FIRST DAY OF THIS YEAR', 'UTC' )->format( 'Y-m-d H:i:s' ) )
									.	"\n AND " . $_CB_database->NameQuote( 'date' ) . " < " . $_CB_database->Quote( Application::Date( 'MIDNIGHT FIRST DAY OF NEXT YEAR', 'UTC' )->format( 'Y-m-d H:i:s' ) );
						break;
					case 'custom':
						$query		.=	"\n AND " . $_CB_database->NameQuote( 'date' ) . " >= " . $_CB_database->Quote( Application::Date( strtoupper( $timeframeStart ), 'UTC' )->format( 'Y-m-d H:i:s' ) )
									.	"\n AND " . $_CB_database->NameQuote( 'date' ) . " < " . $_CB_database->Quote( Application::Date( strtoupper( $timeframeEnd ), 'UTC' )->format( 'Y-m-d H:i:s' ) );
						break;
				}
			}

			$_CB_database->setQuery( $query );

			$cache[$user][$id]		=	(int) $_CB_database->loadResult();
		}

		return $cache[$user][$id];
	}

	/**
	 * Returns an action with auto action specific information
	 *
	 * @param null|UserTable $user
	 * @param array          $variables
	 * @param array          $substitutions
	 * @return ActionInterface
	 */
	public function getAction( ?UserTable $user = null, array $variables = [], array $substitutions = [] ): ActionInterface
	{
		if ( ! $user ) {
			$user				=	\CBuser::getMyInstance()->getUserData();
		}

		$class					=	'\CB\Plugin\AutoActions\Action\\' . ucfirst( $this->getType() ) . 'Action';

		if ( ! class_exists( $class ) ) {
			$action				=	new Action( $this );
		} else {
			$action				=	new $class( $this );
		}

		if ( isset( $variables['self'] ) ) {
			unset( $variables['self'] );
		}

		if ( isset( $variables['user'] ) ) {
			unset( $variables['user'] );
		}

		$substitutions			=	array_merge( $substitutions, CBAutoActions::getExtras( $variables ) );

		$variables['self']		=	$this;
		$variables['user']		=	$user;

		$action->substitutions( $substitutions );
		$action->variables( $variables );

		return $action;
	}

	/**
	 * Triggers the action checking access and conditions then returning its results  (this skips: User)
	 *
	 * @param null|UserTable $user
	 * @param array          $variables
	 * @param array          $substitutions
	 * @return mixed
	 */
	public function triggerAction( ?UserTable $user = null, array $variables = [], array $substitutions = [] )
	{
		if ( ! $user ) {
			$user				=	\CBuser::getMyInstance()->getUserData();
		}

		$action					=	$this->getAction( $user, $variables, $substitutions );
		$return					=	null;

		try {
			$return				=	$action->trigger( $user );

			if ( $action->error() ) {
				$this->setError( implode( '<br />', $action->error() ) );
			}
		} catch ( \Exception $e ) {
			$this->setError( $e->getMessage() );
		}

		return $return;
	}

	/**
	 * Executes the action directly and returns its results (this skips: User, Access, and Conditions)
	 *
	 * @param null|UserTable $user
	 * @return mixed
	 */
	public function executeAction( ?UserTable $user = null )
	{
		if ( ! $user ) {
			$user		=	\CBuser::getMyInstance()->getUserData();
		}

		$action			=	$this->getAction( $user );
		$return			=	null;

		try {
			$return		=	$action->execute( $user );

			if ( $action->error() ) {
				$this->setError( implode( '<br />', $action->error() ) );
			}
		} catch ( \Exception $e ) {
			$this->setError( $e->getMessage() );
		}

		return $return;
	}

	/**
	 * Runs the action with user parsing, variables, conditions, and access checks
	 *
	 * @param array $variables
	 * @return mixed
	 */
	public function runAction( array &$variables = [] )
	{
		global $_CB_database, $_CB_framework, $_PLUGINS;

		static $running						=	[];

		$runningId							=	$this->getId() . ':' . ( $variables['trigger'] ?? 'NONE' );

		if ( \array_key_exists( $runningId, $running ) ) {
			return null;
		}

		$totalUses							=	$this->getParams()->getInt( 'uses_total', 0 );

		if ( $totalUses && ( $this->getUses() >= $totalUses ) ) {
			return null;
		}

		static $connections					=	[];

		$userIds							=	[];
		$users								=	[];
		$loop								=	[];

		switch ( $this->getObject() ) {
			case '0': // Automated
				$user						=	CBAutoActions::getUser( $variables );

				if ( $user->getInt( 'id', 0 ) && $this->getParams()->getBool( 'reload', false ) ) {
					$user->load( $user->getInt( 'id', 0 ) );
				}

				$users[]					=	$user;
				break;
			case '1': // Manually
				$variable					=	'var' . (int) $this->getVariable();

				if ( isset( $variables[$variable] ) ) {
					$variable				=	$variables[$variable];

					if ( \is_string( $variable ) && preg_match( '/^(\d+,\d+)+$/', $variable ) ) {
						$variable			=	cbToArrayOfInt( explode( ',', $variable ) );
					}

					if ( is_iterable( $variable ) ) {
						foreach ( $variable as $varUser ) {
							if ( ( $varUser === '' ) || ( $varUser === null ) || ( $varUser === false ) ) {
								continue;
							}

							$user			=	CBAutoActions::prepareUser( $varUser, [ 'id' ] );

							if ( ! $user instanceof UserTable ) {
								$userIds[]	=	$user;
							} elseif ( $user->getInt( 'id', 0 ) ) {
								$users[]	=	$user;
							}
						}
					} elseif ( ( $variable !== '' ) && ( $variable !== null ) && ( $variable !== false ) ) {
						$user				=	CBAutoActions::prepareUser( $variable );

						if ( ! $user instanceof UserTable ) {
							$userIds[]		=	$user;
						} else {
							if ( $user->getInt( 'id', 0 ) && $this->getParams()->getBool( 'reload', false ) ) {
								$user->load( $user->getInt( 'id', 0 ) );
							}

							$users[]		=	$user;
						}
					}
				}
				break;
			case '2': // Self
				$users[]					=	\CBuser::getMyUserDataInstance();
				break;
			case '3': // Specific
				$objectUsers				=	cbToArrayOfInt( explode( ',', $this->getAction( null, $variables )->string( null, $this->getVariable(), false, false, false, false, true ) ) );

				if ( $objectUsers ) {
					$userIds				=	array_merge( $userIds, $objectUsers );
				}
				break;
			case '4': // Code
				$user						=	CBAutoActions::getUser( $variables );

				$codeOptions				=	explode( '|*|', $this->getParams()->getString( 'object_code_options', 'substitutions' ) );
				$code						=	$this->getAction( $user, $variables )->string( $user, $this->getParams()->getRaw( 'object_code', '' ), false, \in_array( 'translate', $codeOptions, true ), \in_array( 'format', $codeOptions, true ), \in_array( 'content_plugins', $codeOptions, true ), \in_array( 'substitutions', $codeOptions, true ) );

				if ( $code ) {
					try {
						$variable			=	CBAutoActions::outputCode( $code, $this, $user, $variables );
					} catch ( \Exception $e ) {
						$variable			=	null;

						$this->setError( CBTxt::T( 'AUTO_ACTION_USER_CODE_FAILED', ':: Action [action] :: User code failed. Error: [error]', [ '[action]' => $this->getId(), '[error]' => $e->getMessage() ] ) );
					}

					if ( \is_string( $variable ) && preg_match( '/^(\d+,\d+)+$/', $variable ) ) {
						$variable			=	cbToArrayOfInt( explode( ',', $variable ) );
					}

					if ( is_iterable( $variable ) ) {
						foreach ( $variable as $varUser ) {
							if ( ( $varUser === '' ) || ( $varUser === null ) || ( $varUser === false ) ) {
								continue;
							}

							$user			=	CBAutoActions::prepareUser( $varUser );

							if ( ! $user instanceof UserTable ) {
								$userIds[]	=	$user;
							} elseif ( $user->getInt( 'id', 0 ) ) {
								$users[]	=	$user;
							}
						}
					} elseif ( ( $variable !== '' ) && ( $variable !== null ) && ( $variable !== false ) ) {
						$user				=	CBAutoActions::prepareUser( $variable );

						if ( ! $user instanceof UserTable ) {
							$userIds[]		=	$user;
						} else {
							if ( $user->getInt( 'id', 0 ) && $this->getParams()->getBool( 'reload', false ) ) {
								$user->load( $user->getInt( 'id', 0 ) );
							}

							$users[]		=	$user;
						}
					}
				}
				break;
			case '5': // Connections
				$user						=	CBAutoActions::getUser( $variables );
				$userId						=	$user->getInt( 'id', 0 );

				if ( ! isset( $connections[$userId] ) ) {
					$cbConnection			=	new \cbConnection( $userId );

					$connections[$userId]	=	$cbConnection->getActiveConnections( $userId );
				}

				foreach ( $connections[$userId] as $connection ) {
					$userIds[]				=	$connection->id;
				}
				break;
			case '6': // Moderators
				$moderatorViewAccessLevels	=	Application::CmsPermissions()->getGroupsOfViewAccessLevel( Application::Config()->getInt( 'moderator_viewaccesslevel', 3 ), true );

				if ( $moderatorViewAccessLevels ) {
					static $moderators		=	null;

					if ( $moderators === null ) {
						$query				=	'SELECT DISTINCT u.' . $_CB_database->NameQuote( 'id' )
											.	"\n FROM " . $_CB_database->NameQuote( '#__users' ) . " AS u"
											.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
											.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = u.' . $_CB_database->NameQuote( 'id' )
											.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__user_usergroup_map' ) . " AS g"
											.	' ON g.' . $_CB_database->NameQuote( 'user_id' ) . ' = c.' . $_CB_database->NameQuote( 'id' )
											.	"\n WHERE g." . $_CB_database->NameQuote( 'group_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $moderatorViewAccessLevels )
											.	"\n AND u." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	"\n AND c." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	"\n AND c." . $_CB_database->NameQuote( 'approved' ) . " = 1";

						$_CB_database->setQuery( $query );
						$moderators			=	$_CB_database->loadResultArray();
					}

					if ( $moderators ) {
						$userIds			=	array_merge( $userIds, $moderators );
					}
				}
				break;
			case '7': // Query
				$user						=	CBAutoActions::getUser( $variables );

				$queryOptions				=	explode( '|*|', $this->getParams()->getString( 'object_query_options', 'substitutions' ) );
				$query						=	$this->getAction( $user, $variables )->string( $user, $this->getParams()->getRaw( 'object_query', '' ), [ '\CB\Plugin\AutoActions\CBAutoActions', 'escapeSQL' ], \in_array( 'translate', $queryOptions, true ), \in_array( 'format', $queryOptions, true ), \in_array( 'content_plugins', $queryOptions, true ), \in_array( 'substitutions', $queryOptions, true ) );

				if ( $query && ( stripos( trim( $query ), 'SELECT' ) === 0 ) ) {
					try {
						$_CB_database->setQuery( $query );
						$queryUsers			=	$_CB_database->loadResultArray();

						if ( $queryUsers ) {
							$userIds		=	array_merge( $userIds, $queryUsers );
						}
					} catch ( \Exception $e ) {
						$this->setError( CBTxt::T( 'AUTO_ACTION_USER_QUERY_FAILED', ':: Action [action] :: User query failed. Error: [error]', [ '[action]' => $this->getId(), '[error]' => $e->getMessage() ] ) );
					}
				}
				break;
		}

		if ( $this->getParams()->getInt( 'loop', 0 ) ) {
			$loopVariable					=	'var' . $this->getParams()->getInt( 'loop', 0 );

			if ( isset( $variables[$loopVariable] ) ) {
				$loopVariable				=	$variables[$loopVariable];

				if ( is_iterable( $loopVariable ) ) {
					$loop					=	$loopVariable;
				}
			}
		}

		$_PLUGINS->trigger( 'autoactions_onRun', [ &$this, &$variables, &$userIds, &$users, &$loop ] );

		if ( $userIds ) {
			$userIds						=	array_unique( $userIds );

			\CBuser::advanceNoticeOfUsersNeeded( $userIds );

			foreach ( $userIds as $userId ) {
				$user						=	\CBuser::getUserDataInstance( $userId );

				if ( $user->getInt( 'id', 0 ) ) {
					$users[]				=	$user;
				}
			}
		}

		$perUserUses						=	$this->getParams()->getString( 'uses_user', 'custom' );
		$perUserUsesCustom					=	$this->getParams()->getInt( 'uses_user_custom', 0 );
		$return								=	null;

		$running[$runningId]				=	$this->getId();

		foreach ( $users as $user ) {
			if ( $perUserUses && ( $perUserUses !== 'custom' ) ) {
				$limitField					=	\CBuser::getInstance( $user->getInt( 'id', 0 ), false )->getField( $perUserUses, null, 'php', 'none', 'profile', 0, true );

				if ( \is_array( $limitField ) ) {
					$userUses				=	array_shift( $limitField );

					if ( \is_array( $userUses ) ) {
						$userUses			=	implode( '|*|', $userUses );
					}
				} else {
					$userUses				=	$user->getInt( $perUserUses, 0 );
				}

				$userUses					=	(int) $userUses;
			} else {
				$userUses					=	$perUserUsesCustom;
			}

			if ( $userUses && ( $this->getUses( $user ) >= $userUses ) ) {
				continue;
			}

			if ( $user->getInt( 'id', 0 ) ) {
				$password					=	Application::Input()->getString( 'post/passwd', '' );

				if ( ! $password ) {
					$password				=	Application::Input()->getString( 'post/password', '' );
				}
			} else {
				$password					=	'';
			}

			if ( $loop ) {
				foreach ( $loop as $loopKey => $loopVar ) {
					$variables['loop_key']	=	$loopKey;
					$variables['loop']		=	&$loopVar;

					$content				=	$this->triggerAction( $user, $variables, [ 'password' => $password ] );

					if ( ( $return !== '' ) && ( $return !== null ) ) {
						$return				.=	$content;
					} else {
						$return				=	$content;
					}
				}
			} else {
				$content					=	$this->triggerAction( $user, $variables, [ 'password' => $password ] );

				if ( ( $return !== '' ) && ( $return !== null ) ) {
					$return					.=	$content;
				} else {
					$return					=	$content;
				}
			}

			if ( ( ! $this->getError() ) && ( $totalUses || $userUses ) ) {
				$use						=	new UseTable();

				$use->setUserId( $user->getInt( 'id', 0 ) );
				$use->setAutoAction( $this->getId() );

				$use->store();
			}
		}

		unset( $running[$runningId] );

		if ( $this->getError() && $this->getParams()->getBool( 'debug', false ) ) {
			$_CB_framework->enqueueMessage( $this->getError(), 'error' );
		}

		return $return;
	}

	/**
	 * Checks if the action associated with this auto action is even installed
	 *
	 * @return bool
	 */
	public function isActionInstalled(): bool
	{
		return $this->getAction()->installed();
	}

	/**
	 * @return int
	 */
	public function getId(): int
	{
		return $this->getInt( 'id', 0 );
	}

	/**
	 * @param null|int $id
	 */
	public function setId( ?int $id ): void
	{
		$this->id	=	$id;
	}

	/**
	 * @return string
	 */
	public function getSystem(): string
	{
		return $this->getString( 'system', '' );
	}

	/**
	 * @param string $system
	 */
	public function setSystem( string $system ): void
	{
		$this->system	=	$system;
	}

	/**
	 * @return string
	 */
	public function getTitle(): string
	{
		return $this->getString( 'title', '' );
	}

	/**
	 * @param null|string $title
	 */
	public function setTitle( ?string $title ): void
	{
		$this->title	=	$title;
	}

	/**
	 * @return string
	 */
	public function getDescription(): string
	{
		return $this->getString( 'description', '' );
	}

	/**
	 * @param null|string $description
	 */
	public function setDescription( ?string $description ): void
	{
		$this->description	=	$description;
	}

	/**
	 * @return string
	 */
	public function getType(): string
	{
		return $this->getString( 'type', '' );
	}

	/**
	 * @param string $type
	 */
	public function setType( string $type ): void
	{
		$this->type	=	$type;
	}

	/**
	 * @return string
	 */
	public function getTrigger(): string
	{
		return $this->getString( 'trigger', '' );
	}

	/**
	 * @param string $trigger
	 */
	public function setTrigger( string $trigger ): void
	{
		$this->trigger	=	$trigger;
	}

	/**
	 * @return string
	 */
	public function getObject(): string
	{
		return $this->getString( 'object', '0' );
	}

	/**
	 * @param string $object
	 */
	public function setObject( string $object ): void
	{
		$this->object	=	$object;
	}

	/**
	 * @return string
	 */
	public function getVariable(): string
	{
		return $this->getString( 'variable', '' );
	}

	/**
	 * @param null|string $variable
	 */
	public function setVariable( ?string $variable ): void
	{
		$this->variable	=	$variable;
	}

	/**
	 * @return string
	 */
	public function getAccess(): string
	{
		return $this->getString( 'access', '-1' );
	}

	/**
	 * @param null|string $access
	 */
	public function setAccess( ?string $access ): void
	{
		$this->access	=	$access;
	}

	/**
	 * @return Registry
	 */
	public function getConditions(): Registry
	{
		if ( $this->_conditions === null ) {
			$this->_conditions	=	new Registry( $this->conditions );
		}

		return $this->_conditions;
	}

	/**
	 * @param string|array $conditions
	 */
	public function setConditions( $conditions ): void
	{
		$this->getConditions()->load( $conditions );
	}

	/**
	 * @return int
	 */
	public function getPublished(): int
	{
		return $this->getInt( 'published', 0 );
	}

	/**
	 * @param int $published
	 */
	public function setPublished( int $published ): void
	{
		$this->published	=	$published;
	}

	/**
	 * @return int
	 */
	public function getOrdering(): int
	{
		return $this->getInt( 'ordering', 99999 );
	}

	/**
	 * @param int $ordering
	 */
	public function setOrdering( int $ordering ): void
	{
		$this->ordering	=	$ordering;
	}

	/**
	 * @return Registry
	 */
	public function getParams(): Registry
	{
		if ( $this->_params === null ) {
			$this->_params	=	new Registry( $this->params );
		}

		return $this->_params;
	}

	/**
	 * @param string|array $params
	 */
	public function setParams( $params ): void
	{
		$this->getParams()->load( $params );
	}
}