<?php
/**
 * @package     OSL
 * @subpackage  Input
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Input;

use JInput, JFilterInput;

/**
 * Extends JInput class to allow getting raw data from Input object
 *
 * @property    Input $get
 * @property    Input $post
 */
class Input extends JInput
{
	const INPUT_ALLOWRAW = 2;

	const INPUT_ALLOWHTML = 4;

	/**
	 * Constructor.
	 *
	 * @param   array  $source   Source data (Optional, default is $_REQUEST)
	 * @param   array  $options  Array of configuration parameters (Optional)
	 *
	 */
	public function __construct($source = null, array $options = [])
	{
		if ($source instanceof JInput)
		{
			$reflection = new \ReflectionClass($source);
			$property   = $reflection->getProperty('data');
			$property->setAccessible(true);
			$source = $property->getValue($source);
		}

		if (!isset($options['filter']))
		{
			//Set default filter so that getHtml can be returned properly
			if (version_compare(JVERSION, '4.0.0-dev', 'ge'))
			{
				//Set default filter so that getHtml can be returned properly
				$options['filter'] = JFilterInput::getInstance([], [], 1, 1);
			}
			else
			{
				$options['filter'] = JFilterInput::getInstance(null, null, 1, 1);
			}
		}

		parent::__construct($source, $options);
	}

	/**
	 *
	 * Get data from the input
	 *
	 * @param   int  $mask
	 *
	 * @return array
	 */
	public function getData($mask = Input::INPUT_ALLOWHTML)
	{
		if ($mask & 2)
		{
			return $this->data;
		}

		return $this->filter->clean($this->data, null);
	}

	/**
	 * Set data for the input object. This is usually called when you get data, modify it, and then set it back
	 *
	 * @param $data
	 */
	public function setData(array $data)
	{
		$this->data = $data;
	}

	/**
	 * Check to see if a variable is available in the input or not
	 *
	 * @param   string  $name  the variable name
	 *
	 * @return boolean
	 */
	public function has($name)
	{
		if (isset($this->data[$name]))
		{
			return true;
		}

		return false;
	}

	/**
	 * Remove a variable from input
	 *
	 * @param $name
	 */
	public function remove($name)
	{
		if (isset($this->data[$name]))
		{
			unset($this->data[$name]);
		}
	}

	/**
	 * Magic method to get an input object
	 *
	 * @param   mixed  $name  Name of the input object to retrieve.
	 *
	 * @return  Input  The request input object
	 */
	public function __get($name)
	{
		if (isset($this->inputs[$name]))
		{
			return $this->inputs[$name];
		}

		$className = 'JInput' . ucfirst($name);

		if (class_exists($className))
		{
			$this->inputs[$name] = new $className(null, $this->options);

			return $this->inputs[$name];
		}

		$superGlobal = '_' . strtoupper($name);

		if (isset($GLOBALS[$superGlobal]))
		{
			$this->inputs[$name] = new Input($GLOBALS[$superGlobal], $this->options);

			return $this->inputs[$name];
		}
	}
}