<?php
/**
 * @package	RSMembership!
 * @copyright	(c) 2013 - 2018 RSJoomla!
 * @link		https://www.rsjoomla.com
 * @license	GNU General Public License http://www.gnu.org/licenses/gpl-3.0.en.html
 */

// No direct access.
defined('_JEXEC') or die('Restricted access');

use Joomla\Utilities\ArrayHelper;

JLoader::register('PrivacyPlugin', JPATH_ADMINISTRATOR . '/components/com_privacy/helpers/plugin.php');
JLoader::register('PrivacyRemovalStatus', JPATH_ADMINISTRATOR . '/components/com_privacy/helpers/removal/status.php');


/**
 * RSMembership! Privacy Plugin.
 */
class PlgPrivacyRsmembership extends PrivacyPlugin
{
	const EXTENSION = 'plg_privacy_rsmembership';

	/**
	 * Can the plugin run?
	 *
	 * @access public
	 *
	 * @return bool
	 */
	public function canRun()
	{
		return file_exists(JPATH_ADMINISTRATOR . '/components/com_rsmembership/helpers/helper.php');
	}

    /**
     * Performs validation to determine if the data associated with a remove information request can be processed
     *
     * This event will not allow a super user account to be removed
     *
     * @param   PrivacyTableRequest  $request  The request record being processed
     * @param   JUser                $user     The user account associated with this request if available
     *
     * @return  PrivacyRemovalStatus
     *
     * @since   3.9.0
     */
    public function onPrivacyCanRemoveData(PrivacyTableRequest $request, JUser $user = null)
    {
        $status = new PrivacyRemovalStatus;

        if (!$user)
        {
            return $status;
        }

        if ($user->authorise('core.admin'))
        {
            $status->canRemove = false;
            $status->reason    = JText::_('PLG_PRIVACY_RSMEMBERSHIP_ERROR_CANNOT_REMOVE_SUPER_USER');
        }

        return $status;
    }

	/**
	 * Function that retrieves the information for the RSMembership! Component Capabilities
	 * @return array
	 *
	 */
	public function onPrivacyCollectAdminCapabilities()
	{
		if (!$this->canRun())
		{
			return array();
		}

		$capabilities = array(
			JText::_('PLG_PRIVACY_RSMEMBERSHIP_CAPABILITIES_GENERAL') => array(
				JText::_('PLG_PRIVACY_RSMEMBERSHIP_CAPABILITIES_SUBSCRIPTIONS'),
				JText::_('PLG_PRIVACY_RSMEMBERSHIP_CAPABILITIES_TRANSACTIONS'),
				JText::_('PLG_PRIVACY_RSMEMBERSHIP_CAPABILITIES_RECAPTCHA'),
			)
		);

		return $capabilities;
	}
	/**
	 *
	 * @param   PrivacyTableRequest  $request  The request record being processed
	 * @param   JUser                $user     The user account associated with this request if available
	 *
	 * @return  PrivacyExportDomain[]
	 *
	 * @since   3.9.0
	 */
	public function onPrivacyExportRequest(PrivacyTableRequest $request, JUser $user = null)
	{
		if (!$this->canRun())
		{
			return array();
		}

		if (!$user)
		{
			return array();
		}

        require_once JPATH_ADMINISTRATOR . '/components/com_rsmembership/helpers/helper.php';

		/** @var JTableUser $userTable */
		$userTable = JUser::getTable();
		$userTable->load($user->id);

		$domains = array();
		$domains[] = $this->createSubscriberCustomFields($userTable);
		$domains[] = $this->createTransactions($userTable);
		$domains[] = $this->createSubscriptions($userTable);
		$domains[] = $this->createAccessLogs($userTable);

		return $domains;
	}

	/**
	 * Removes the data associated with a remove information request
	 *
	 * This event will pseudoanonymise the user account
	 *
	 * @param   PrivacyTableRequest  $request  The request record being processed
	 * @param   JUser                $user     The user account associated with this request if available
	 *
	 * @return  void
	 *
	 * @since   3.9.0
	 */
	public function onPrivacyRemoveData(PrivacyTableRequest $request, JUser $user = null)
	{
		if (!$this->canRun())
		{
			return;
		}

		// This plugin only processes data for registered user accounts
		if (!$user)
		{
			return;
		}

        // Anonymise data
        require_once JPATH_ADMINISTRATOR . '/components/com_rsmembership/helpers/helper.php';
        RSMembershipHelper::anonymise($user->id, 0);
	}

    /**
     * Create the domain for the RSMembership! subscriber custom fields
     *
     * @param   JTableUser  $user  The JTableUser object to process
     *
     * @return  PrivacyExportDomain
     *
     * @since   3.9.0
     */
    private function createSubscriberCustomFields(JTableUser $user)
    {
        $domain = $this->createDomain('user_rsmembership_subscriber_fields', 'rsmembership_subscriber_fields');

        if ($fields = RSMembershipHelper::getUserFields($user->id))
        {
            $domain->addItem($this->createItemFromArray($fields));
        }

        return $domain;
    }

	/**
	 * Create the domain for the transactions list
	 *
	 * @param   JTableUser  $user  The JTableUser object to process
	 *
	 * @return  PrivacyExportDomain
	 *
	 * @since   3.9.0
	 */
	private function createTransactions(JTableUser $user)
	{
		$domain = $this->createDomain('user_rsmembership_transactions', 'rsmembership_transactions');

		$query = $this->db->getQuery(true)
			->select('*')
			->from($this->db->qn('#__rsmembership_transactions'))
			->where($this->db->qn('user_id') . ' = '. $this->db->quote($user->id));

		if ($items = $this->db->setQuery($query)->loadAssocList())
        {
            foreach (array('user_id', 'id', 'hash', 'custom', 'response_log') as $column)
            {
                $items = ArrayHelper::dropColumn($items, $column);
            }
            foreach ($items as $item)
            {
                $domain->addItem($this->createItemFromArray($item));
            }
        }

		return $domain;
	}

    /**
     * Create the domain for the access logs
     *
     * @param   JTableUser  $user  The JTableUser object to process
     *
     * @return  PrivacyExportDomain
     *
     * @since   3.9.0
     */
    private function createAccessLogs(JTableUser $user)
    {
        $domain = $this->createDomain('user_rsmembership_access_logs', 'rsmembership_access_logs');

        $query = $this->db->getQuery(true)
            ->select('*')
            ->from($this->db->qn('#__rsmembership_logs'))
            ->where($this->db->qn('user_id') . ' = '. $this->db->quote($user->id));

        if ($items = $this->db->setQuery($query)->loadAssocList())
        {
            // Remove user ID and ID columns
			foreach (array('user_id', 'id') as $column)
			{
				$items = ArrayHelper::dropColumn($items, $column);
			}
            foreach ($items as $item)
            {
                $domain->addItem($this->createItemFromArray($item));
            }
        }

        return $domain;
    }

    /**
     * Create the domain for the subscriptions
     *
     * @param   JTableUser  $user  The JTableUser object to process
     *
     * @return  PrivacyExportDomain
     *
     * @since   3.9.0
     */
    private function createSubscriptions(JTableUser $user)
    {
        $domain = $this->createDomain('user_rsmembership_subscriptions', 'rsmembership_subscriptions');

        // Get the database object
        $db 	= &$this->db;
        $query	= $db->getQuery(true);

        $query
            ->select($db->qn('m.name'))
            ->select('IFNULL(' . $db->qn('c.name') . ', ' . $db->q(JText::_('COM_RSMEMBERSHIP_NO_CATEGORY')) . ') AS ' . $db->qn('category_name'))
            ->select($db->qn('ms.extras'))
            ->select($db->qn('ms.status'))
            ->select($db->qn('ms.notified'))
            ->select($db->qn('ms.membership_start'))
            ->select($db->qn('ms.membership_end'))
            ->select($db->qn('ms.published'))
            ->from($db->qn('#__rsmembership_membership_subscribers', 'ms'))
            ->where($db->qn('ms.user_id') . ' = ' . $db->q($user->id))
            ->join('left', $db->qn('#__rsmembership_memberships', 'm').' ON '.$db->qn('ms.membership_id').' = '.$db->qn('m.id'))
            ->join('left',$db->qn('#__rsmembership_categories', 'c').' ON '.$db->qn('m.category_id').' = '.$db->qn('c.id'));

        if ($items = $db->setQuery($query)->loadAssocList())
        {
            foreach ($items as $item)
            {
                $item['status'] = JText::_('COM_RSMEMBERSHIP_STATUS_' . $item['status']);
                if ($item['extras'])
                {
                    $item['extras']	= RSMembershipHelper::getExtrasNames($item['extras']);
                }
                $domain->addItem($this->createItemFromArray($item));
            }
        }

        return $domain;
    }
}